import { useState, useEffect, useRef } from "react";
import { useDispatch, useSelector } from "react-redux";
import Image from "next/image";
import io from "socket.io-client";
import { addMessage, setMessages } from "@/store/slices/chatSlice";
import { fetchSendMessage, fetchReceiveMessage } from "@/utils/apiActions";
import user from "../../../public/images/user1.png";
import "./userchat.scss";
import Link from "next/link";

const UserChat = () => {
    const [loading, setLoading] = useState(true);
    const dispatch = useDispatch();
    const messages = useSelector((state) => state.chat.messages);
    const [message, setMessage] = useState('');
    const socketRef = useRef(null);
    const chatBodyRef = useRef(null);
    const [isOpen, setIsOpen] = useState(false);
    // const dropdownRef = useRef(null);

    // const toggleDropdown = (event) => {
    //     event.stopPropagation();
    //     setIsOpen((prev) => !prev);
    // };

    // useEffect(() => {
    //     const handleClickOutside = (event) => {
    //         if (dropdownRef.current && !dropdownRef.current.contains(event.target)) {
    //             setIsOpen(false);
    //         }
    //     };

    //     document.addEventListener("click", handleClickOutside);
    //     return () => {
    //         document.removeEventListener("click", handleClickOutside);
    //     };
    // }, []);
    
    useEffect(() => {
        socketRef.current = io(process.env.NEXT_PUBLIC_SOCKET_URL);
        console.log("WebSocket connected:", socketRef.current);

        const loadMessages = async () => {
            try {
                const response = await fetchReceiveMessage();
                console.log("Fetched Messages Response:", response.data);
                if (response.data) {
                    dispatch(setMessages(response.data));
                }
            } catch (error) {
                console.error("Error fetching messages:", error);
            } finally {
                setLoading(false);
            }
        };

        loadMessages();

        const handleReceiveMessage = (newMessage) => {
            dispatch(addMessage(newMessage));
        };

        socketRef.current.on("receiveMessage", handleReceiveMessage);

        return () => {
            if (socketRef.current) {
                socketRef.current.off("receiveMessage", handleReceiveMessage);
                socketRef.current.disconnect();
            }
        };
    }, [dispatch, process.env.NEXT_PUBLIC_SOCKET_URL]);

    useEffect(() => {
        if (chatBodyRef.current) {
            setTimeout(() => {
                chatBodyRef.current.scrollTop = chatBodyRef.current.scrollHeight;
            }, 100);
        }
    }, [messages]);

    const sendMessage = async () => {
        if (!message.trim()) {
            console.warn("Message is empty, not sending.");
            return;
        }
    
        try {
            const receiverId = "199";
            console.log("Sending message:", { receiverId, message, });

            const response = await fetchSendMessage(receiverId, message, null);
            if (!response) {
                console.error("No response received from fetchSendMessage.");
                return;
            }
    
            if (response?.success) {
                const newMessage = { 
                    sender: "You", 
                    text: message, 
                    time: new Date().toLocaleTimeString() 
                };
    
                console.log("Emitting socket message:", newMessage);
                if (socketRef.current) {
                    socketRef.current.emit("sendMessage", newMessage);
                } else {
                    console.error("socketRef.current is null, socket may not be initialized.");
                }
    
                dispatch(addMessage(newMessage));
                setMessage(""); 
                document.getElementById("chatTextarea").value = ""; 
            } else {
                console.error("Message send failed:", response?.error || "Unknown error");
            }
        } catch (error) {
            console.error("Error sending message:", error);
        }
    };


    const formatDateHeader = (dateString) => {
        const messageDate = new Date(dateString);
        const today = new Date();
        const yesterday = new Date();
        yesterday.setDate(yesterday.getDate() - 1);
    
        if (
            messageDate.toDateString() === today.toDateString()
        ) {
            return "Today";
        } else if (
            messageDate.toDateString() === yesterday.toDateString()
        ) {
            return "Yesterday";
        } else {
            return messageDate.toLocaleDateString(undefined, {
                month: "long",
                day: "numeric",
            });
        }
    };
    
    
    return (
        <div className="card shadow-lg border-0 chat-boxes rounded-4 justify-content-between position-relative">
            {loading ? (
                <div className="loader_bg">
                    <span className="loader my-3 mx-auto d-block"></span>
                </div>
            ) : (
                <>
                    {messages.length > 0 && messages[messages.length - 1].receiver && (
                        <div className="card-header p-3 bg-primary text-white rounded-top-4">
                            <h5 className="card-title mb-0 fw-bold text-uppercase">
                                {messages[messages.length - 1].receiver.name} {messages[messages.length - 1].receiver.lastname}
                            </h5>
                        </div>
                    )}

                    <div ref={chatBodyRef} className="card-body gap-2 py-4 overflow-y-auto" style={{ maxHeight: "400px", position: "relative" }}>
                        {messages.length > 0 ? (
                            messages.reduce((acc, msg, index) => {
                                const messageDate = new Date(msg.created_at);
                                const formattedDate = formatDateHeader(messageDate);

                                if (index === 0 || formatDateHeader(messages[index - 1].created_at) !== formattedDate) {
                                    acc.push(
                                        <div key={`date-${index}`} className="date-header text-center text-muted sticky-top bg-light p-1 rounded">
                                            {formattedDate}
                                        </div>
                                    );
                                }

                                acc.push(
                                    <div key={index} className={msg.sender_id === 199 ? "right-message" : "left-message"}>
                                        <div className="d-flex gap-2">
                                            {msg.sender_id !== 199 && (
                                                <Image src={msg.receiver?.photo?.original_url || user} width={40} height={40} alt="" className="rounded-pill object-fit-cover" />
                                            )}
                                            <div>
                                                <h6>{msg.name} {msg.lastname}</h6>
                                                <p className="date_time">{new Date(msg.created_at).toLocaleTimeString()}</p>
                                                <p className="bg-primary p-2 text-white rounded-3 my-1 px-3">{msg.message}</p>
                                            </div>
                                        </div>
                                        <div className="right-message">
                                            <div className="d-flex gap-2 justify-content-end">
                                                <div className="text-end">
                                                    <p className="date_time">{new Date(msg.created_at).toLocaleTimeString()}</p>
                                                    <p className="bg-grey p-2 text-black rounded-3 my-1 text-start px-3">
                                                        Lorem Ipsum is simply dummy text of the printing and typesetting industry.
                                                    </p>
                                                </div>
                                                {/* <div ref={dropdownRef}>
                                                    <button className="btn btn-light rounded-circle p-1 select_dropdown" onClick={toggleDropdown}>
                                                        <svg xmlns="http://www.w3.org/2000/svg" height="30px" viewBox="0 -960 960 960" width="30px" fill="#1f1f1f">
                                                            <path d="M480-160q-33 0-56.5-23.5T400-240q0-33 23.5-56.5T480-320q33 0 56.5 23.5T560-240q0 33-23.5 56.5T480-160Zm0-240q-33 0-56.5-23.5T400-480q0-33 23.5-56.5T480-560q33 0 56.5 23.5T560-480q0 33-23.5 56.5T480-400Zm0-240q-33 0-56.5-23.5T400-720q0-33 23.5-56.5T480-800q33 0 56.5 23.5T560-720q0 33-23.5 56.5T480-640Z"/>
                                                        </svg>
                                                    </button>

                                                    <div className={`dropdown-menu ${isOpen ? "show" : ""}`}>
                                                        <button className="dropdown-item d-flex align-items-center" onClick={() => alert("Forwarding...")}>
                                                            <svg xmlns="http://www.w3.org/2000/svg" height="24px" viewBox="0 -960 960 960" width="24px" fill="#1f1f1f"><path d="m640-280-57-56 184-184-184-184 57-56 240 240-240 240ZM80-200v-160q0-83 58.5-141.5T280-560h247L383-704l57-56 240 240-240 240-57-56 144-144H280q-50 0-85 35t-35 85v160H80Z"/></svg>
                                                            <span className="ms-2">Forward</span>
                                                        </button>
                                                        <button className="dropdown-item d-flex align-items-center text-danger" onClick={() => alert("Deleting...")}>
                                                            <svg xmlns="http://www.w3.org/2000/svg" height="24px" viewBox="0 -960 960 960" width="24px" fill="#1f1f1f"><path d="M280-120q-33 0-56.5-23.5T200-200v-520h-40v-80h200v-40h240v40h200v80h-40v520q0 33-23.5 56.5T680-120H280Zm80-160h80v-360h-80v360Zm160 0h80v-360h-80v360Z"/></svg>
                                                            <span className="ms-2">Delete</span>
                                                        </button>
                                                    </div>
                                                </div> */}
                                            </div>
                                        </div>
                                    </div>
                                );
                                return acc;
                            }, [])
                        ) : (
                            <p className="text-center text-gray-500">No messages yet.</p>
                        )}
                    </div>
                </>
            )}

            <div className="card-footer gap-2 py-3">
                <div className="row align-items-center">
                    <div className="col-md-1 p-0 text-center">
                        <div className="bg-primary rounded-pill file-ic">
                            <svg width="30" height="30" viewBox="0 0 30 30" fill="none" xmlns="http://www.w3.org/2000/svg">
                                <path d="M5.66992 14.3313L14.2874 5.71255C14.8956 5.09749 15.6193 4.60872 16.417 4.27435C17.2147 3.93998 18.0707 3.7666 18.9356 3.76419C19.8006 3.76177 20.6575 3.93037 21.457 4.26029C22.2566 4.5902 22.9831 5.07493 23.5946 5.68658C24.2062 6.29824 24.6908 7.02476 25.0206 7.82437C25.3505 8.62398 25.5189 9.4809 25.5164 10.3458C25.5139 11.2108 25.3404 12.0667 25.0059 12.8644C24.6714 13.6621 24.1826 14.3857 23.5674 14.9938L13.6249 24.9375C12.8044 25.7581 11.6916 26.219 10.5312 26.219C9.37079 26.219 8.25794 25.7581 7.43742 24.9375C6.61691 24.117 6.15595 23.0042 6.15595 21.8438C6.15595 20.6834 6.61691 19.5706 7.43742 18.75L17.3824 8.8063C17.7951 8.40791 18.3477 8.18753 18.9212 8.19263C19.4948 8.19773 20.0434 8.4279 20.4489 8.83356C20.8544 9.23922 21.0843 9.78792 21.0892 10.3615C21.0941 10.935 20.8735 11.4876 20.4749 11.9L11.8562 20.5175" stroke="white" strokeWidth="2" strokeLinecap="round" className="round"/>
                            </svg>
                        </div>
                    </div>
                    <div className="col-md-10">
                        <div className="form-group">
                        <textarea
                            id="chatTextarea"
                            className="form-control"
                            placeholder="Type here"
                            rows="1"
                            value={message}
                            onChange={(e) => setMessage(e.target.value)}
                            onKeyDown={(e) => {
                                if (e.key === "Enter" && !e.shiftKey) {
                                    e.preventDefault();
                                    sendMessage();
                                }
                            }}
                        ></textarea>

                        </div>
                    </div>
                    <div className="col-md-1 p-0 text-center">
                        <button onClick={sendMessage} type="submit" className="border-0 bg-transparent">
                            <svg width="45" height="45" viewBox="0 0 36 36" fill="none" xmlns="http://www.w3.org/2000/svg">
                                <circle cx="18" cy="18" r="18" fill="#FF3158"/>
                                <g clipPath="url(#clip0_3984_3927)">
                                <path d="M26.5917 18.1606C26.6803 18.1198 26.755 18.0538 26.8064 17.9708C26.8579 17.8879 26.8839 17.7917 26.8812 17.6942C26.8785 17.5966 26.8472 17.502 26.7913 17.4221C26.7353 17.3421 26.6571 17.2803 26.5664 17.2444L12.0427 11.4896L12.042 11.4889L11.5912 11.3095C11.5058 11.2756 11.4127 11.2659 11.3221 11.2816C11.2316 11.2972 11.1471 11.3376 11.0781 11.3983C11.009 11.4589 10.9581 11.5375 10.9309 11.6253C10.9037 11.7131 10.9013 11.8067 10.924 11.8957L11.0427 12.365L11.0421 12.3672L12.4818 18.092L11.3604 23.8877L11.3605 23.8905L11.2678 24.3656C11.2504 24.4557 11.2582 24.5488 11.2903 24.6347C11.3225 24.7206 11.3777 24.796 11.4499 24.8526C11.5221 24.9092 11.6086 24.9447 11.6997 24.9553C11.7908 24.9659 11.8831 24.9512 11.9664 24.9127L26.5917 18.1606ZM23.9689 18.2733L13.4055 18.5652L13.4814 18.173C13.4954 18.1011 13.4933 18.027 13.4754 17.956L13.378 17.5686L23.9413 17.2767L25.1173 17.7429L23.9696 18.274L23.9689 18.2733Z" fill="#fff"/>
                                </g>
                                <defs>
                                <clipPath id="clip0_3984_3927">
                                <rect width="15.9537" height="15.9537" fill="white" transform="translate(15.4996 6.72339) rotate(43.4174)"/>
                                </clipPath>
                                </defs>
                            </svg>
                        </button>
                    </div>
                </div>
            </div>
        </div>
    );
};

export default UserChat;
