"use client";

import { useState } from "react";
import { useRouter } from "next/navigation";
import { useForm } from "react-hook-form";
import * as yup from 'yup';
import InputField from "@/components/FormFields/InputField";
import PasswordField from "@/components/FormFields/PasswordField";
import { yupResolver } from '@hookform/resolvers/yup';
import Link from "next/link";

const schema = yup.object({
    fname: yup.string().required('First name is required.'),
    lname: yup.string().required('Last name is required.'),
    email: yup.string().email('Invalid Email').required('Please Enter your email').matches(/^(([^<>()\[\]\\.,;:\s@"]+(\.[^<>()\[\]\\.,;:\s@"]+)*)|(".+"))@((\[[0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}])|(([a-zA-Z\-0-9]+\.)+[a-zA-Z]{2,}))$/, "Invalid Email"),
    password: yup.string().required('Password is required.').matches(
        /^(?=.*[a-z])(?=.*[A-Z])(?=.*[0-9])(?=.{8,})/,
        "Password must contain at least 8 characters, including one uppercase letter, one lowercase letter, and one number."
    ),
    conf_password: yup.string().oneOf([yup.ref('password'), null], 'Passwords must match'),
}).required();

const SignUpForm=()=>{
    const [loading, setLoading] = useState(false);
    const [errorMessage, setErrorMessage] = useState(null);
    const router = useRouter();
    const { register, handleSubmit, formState: { errors } } = useForm({
        resolver: yupResolver(schema)
    });

    const onSubmit = async (data) => {
        setErrorMessage(null);
        setLoading(true);

        // Prepare form data
        const formData = new FormData();
        formData.append("firstname", data.fname);
        formData.append("lastname", data.lname);
        formData.append("email", data.email);
        formData.append("password", data.password);
        formData.append("roles[]", 2);

        // Make the fetch request
        const response = await fetch(`${process.env.NEXT_PUBLIC_API_URL}api/register`, {
            method: "POST",
            body: formData,
        });

        setLoading(false);

        const result = await response.json();
        if(result.status === 1){
            router.push('/auth/thankyou');
        }
        else{
            setErrorMessage(result.message);
        }
    }

    return(
        <>
        <section className="signup-form-section">
            <div className="container">
                <div className="signup-form shadow rounded bg-white mx-auto p-3 p-sm-4">
                    <form onSubmit={handleSubmit(onSubmit)} className={`row`}>
                        <div className="form-group mb-20 col-md-6">
                            <InputField
                                register={register}
                                disabled={loading}
                                fieldClass={'form-control shadow-none'}
                                hideLabel={false}
                                labelClass={'text-black-200 h6 font-weight-600 mb-2'}
                                labelText={'First Name:'}
                                placeHolder={'First Name'}
                                onFieldChange={(e)=>{}}
                                required={true}
                                errors={errors}
                                fieldType={"text"}
                                fieldId={"fname"}
                                fieldName={'fname'}
                                autoComplete={'fname'}
                            />
                        </div>
                        <div className="form-group mb-20 col-md-6">
                            <InputField
                                register={register}
                                disabled={loading}
                                fieldClass={'form-control shadow-none'}
                                hideLabel={false}
                                labelClass={'text-black-200 h6 font-weight-600 mb-2'}
                                labelText={'Last Name:'}
                                placeHolder={'Last Name'}
                                onFieldChange={(e)=>{}}
                                required={true}
                                errors={errors}
                                fieldType={"text"}
                                fieldId={"lname"}
                                fieldName={'lname'}
                                autoComplete={'lname'}
                            />
                        </div>
                        <div className="form-group mb-20 col-12">
                            <InputField
                                register={register}
                                disabled={loading}
                                fieldClass={'form-control shadow-none'}
                                hideLabel={false}
                                labelClass={'text-black-200 h6 font-weight-600 mb-2'}
                                labelText={'Email Address:'}
                                placeHolder={'Email Address'}
                                onFieldChange={(e)=>{}}
                                required={true}
                                errors={errors}
                                fieldType={"text"}
                                fieldId={"email"}
                                fieldName={'email'}
                                autoComplete={'email'}
                            />
                        </div>
                        <div className="form-group mb-20 col-12">
                            <PasswordField
                                register={register}
                                disabled={loading}
                                fieldClass={'form-control shadow-none'}
                                fieldName={'password'}
                                labelClass={'text-black-200 h6 font-weight-600 mb-2'}
                                fieldId={'password'}
                                hideLabel={false}
                                labelText={'Password:'}
                                placeHolder={'Password'}
                                required={true}
                                errors={errors}
                                visibleTrigger={true}
                            />
                        </div>
                        <div className="form-group mb-20 col-12">
                            <PasswordField
                                register={register}
                                disabled={loading}
                                fieldClass={'form-control shadow-none'}
                                fieldName={'conf_password'}
                                labelClass={'text-black-200 h6 font-weight-600 mb-2'}
                                fieldId={'conf_password'}
                                hideLabel={false}
                                labelText={'Retype Password:'}
                                placeHolder={'Retype Password'}
                                required={true}
                                errors={errors}
                                visibleTrigger={true}
                            />
                        </div>
                        <p className="mb-3 text-center">By creating an account you agree to our <Link href="/terms-of-use" target="_blank">Terms of Use </Link> & <Link href="/privacy-policy" target="_blank">Privacy Policy </Link>.</p>
                        <div className="form-group col-12">
                            {!loading ?
                                <button className="btn btn-primary w-100" type="submit">Sign Up</button>
                            :
                                <div className="btn btn-primary w-100">Updating...</div>
                            }
                        </div>
                    </form>
                    {errorMessage!==null &&
                        <div className="alert alert-danger mt-2" role="alert">{errorMessage}</div>
                    }
                </div>
            </div>
        </section>
        </>
    )
}

export default SignUpForm;