"use client";

import { useState, useEffect } from "react";
import { useRouter, useSearchParams } from "next/navigation";
import { useForm } from "react-hook-form";
import * as yup from 'yup';
import { fetchUserDetail } from '@/utils/apiActions';
import InputField from "@/components/FormFields/InputField";
import { yupResolver } from '@hookform/resolvers/yup';

const schema = yup.object({
    email: yup.string().email('Invalid email').required('Email is required'),
    password: yup.string().min(6, 'Password must be at least 6 characters').required('Please enter your new password'),
    passwordConfirmation: yup.string().oneOf([yup.ref('password'), null], 'Passwords must match').required('Please confirm your password'),
}).required();

const ForgotPasswordForm = () => {
    const [loading, setLoading] = useState(false);
    const [errorMessage, setErrorMessage] = useState(null);
    const [successMessage, setSuccessMessage] = useState(null);
    const router = useRouter();
    const searchParams = useSearchParams();
    const tokenFromURL = searchParams.get("token");
    const [showPassword, setShowPassword] = useState(false);
    const [showPassword1, setShowPassword1] = useState(false);
    const handleClickShowPassword = () => {
        setShowPassword(!showPassword);
    };
    const handleClickShowPassword1 = () => {
        setShowPassword1(!showPassword1);
    };
    useEffect(() => {
        if (!tokenFromURL) {
            router.push("/auth/login/");
        }
    }, [tokenFromURL, router]);

    const { register, handleSubmit, formState: { errors }, setValue } = useForm({
        resolver: yupResolver(schema)
    });

    const onSubmit = async (data) => {
        try {
            setErrorMessage(null);
            setSuccessMessage(null);
            setLoading(true);
    
            const formData = new FormData();
            formData.append("email", data.email);
            formData.append("password", data.password);
            formData.append("password_confirmation", data.passwordConfirmation);
            formData.append("token", tokenFromURL);
    
            const response = await fetch(`${process.env.NEXT_PUBLIC_API_URL}api/reset-password`, {
                method: "POST",
                body: formData,
            });
    
            const result = await response.json();
    
            if (response === 0) {
                setErrorMessage(result.error || "An error occurred. Please try again.");
            } else {
                setSuccessMessage(result.message || "Password reset successful!");
                setTimeout(() => {
                    router.push("/auth/login/");
                }, 3000);
            }
        } catch (error) {
            setErrorMessage("Something went wrong. Please try again later.");
        } finally {
            setLoading(false);
        }
    };    
    useEffect(() => {
        if (tokenFromURL) {
            const emailFromURL = searchParams.get("email");
            if (emailFromURL) {
                setValue("email", emailFromURL);
            }
        }
    }, [tokenFromURL, searchParams, setValue]);
    
    if (!tokenFromURL) {
        return null;
    }

    return (
        <section className="login-form-section">
            <div className="container">
                <div className="login-form shadow rounded bg-white mx-auto p-3 p-sm-4">
                    {errorMessage && <div className="alert alert-danger text-center" role="alert">{errorMessage}</div>}
                    {successMessage && <div className="alert alert-success text-center" role="alert">{successMessage}</div>}
                    <form onSubmit={handleSubmit(onSubmit)} className="row">
                        <div className="form-group mb-20 col-12 position-relative">
                            <InputField
                                register={register}
                                disabled={loading}
                                fieldClass="form-control shadow-none"
                                hideLabel={false}
                                labelClass="text-black-200 h6 font-weight-600 mb-2"
                                labelText="Email:"
                                placeHolder="Email"
                                errors={errors}
                                fieldId="email"
                                fieldName="email"
                                autoComplete=""
                                readOnly={true}
                            />
                        </div>
                        <div className="form-group mb-20 col-12 position-relative">
                            <InputField
                                register={register}
                                disabled={loading}
                                fieldClass="form-control shadow-none"
                                hideLabel={false}
                                labelClass="text-black-200 h6 font-weight-600 mb-2"
                                labelText="Password:"
                                placeHolder="Password"
                                errors={errors}
                                fieldType={showPassword ? "text" : "password"}
                                fieldId="password"
                                fieldName="password"
                                type={showPassword ? "text" : "password"}
                                aria-invalid={errors.name ? "true" : "false"}
                                autoComplete=""
                            />
                            <span className="show-password-trigger mt-3 me-2" onClick={handleClickShowPassword}>
                                {showPassword ?
                                <>
                                <svg width="24" height="24" viewBox="0 0 36 36" fill="none" xmlns="http://www.w3.org/2000/svg">
                                    <g>
                                        <path d="M25.1901 20.3996C25.4773 19.6322 25.623 18.8191 25.6201 17.9996C25.6201 16.1803 24.8974 14.4354 23.6109 13.1489C22.3244 11.8624 20.5795 11.1396 18.7601 11.1396C17.9508 11.1406 17.1482 11.2862 16.3901 11.5696L18.0001 13.2296C18.2449 13.1905 18.4923 13.1704 18.7401 13.1696C20.0266 13.167 21.2618 13.6734 22.1761 14.5783C23.0905 15.4833 23.6096 16.7133 23.6201 17.9996C23.6193 18.2475 23.5993 18.4949 23.5601 18.7396L25.1901 20.3996Z" fill="black" />
                                        <path d="M34.29 17.5303C30.92 11.3003 25.01 7.53027 18.47 7.53027C16.6893 7.53445 14.9206 7.82136 13.23 8.38027L14.84 10.0003C16.0258 9.6938 17.2452 9.53593 18.47 9.53027C24.1 9.53027 29.22 12.6703 32.27 17.9603C31.1511 19.9229 29.6679 21.6539 27.9 23.0603L29.32 24.4803C31.3661 22.8296 33.0653 20.7905 34.32 18.4803L34.58 18.0003L34.29 17.5303Z" fill="black" />
                                        <path d="M4.86988 5.78004L9.32988 10.24C6.5108 12.0552 4.20678 14.5658 2.63988 17.53L2.37988 18L2.63988 18.48C6.00988 24.71 11.9199 28.48 18.4599 28.48C21.0125 28.4795 23.532 27.9018 25.8299 26.79L30.8299 31.79L32.5799 30.29L6.57988 4.29004L4.86988 5.78004ZM14.6199 15.53L21.2699 22.18C20.5185 22.6451 19.6535 22.8942 18.7699 22.9C18.1278 22.9001 17.4921 22.7731 16.8993 22.5265C16.3064 22.2799 15.7682 21.9185 15.3156 21.4631C14.863 21.0077 14.5049 20.4672 14.2619 19.8729C14.019 19.2786 13.8959 18.6421 13.8999 18C13.9112 17.1266 14.1601 16.2727 14.6199 15.53ZM13.1699 14.08C12.2343 15.3994 11.7958 17.0073 11.932 18.619C12.0682 20.2307 12.7703 21.7422 13.914 22.8859C15.0577 24.0296 16.5692 24.7317 18.1809 24.8679C19.7926 25.0041 21.4005 24.5656 22.7199 23.63L24.3199 25.23C22.4677 26.0218 20.4742 26.43 18.4599 26.43C12.8299 26.43 7.70988 23.29 4.65988 18C6.12359 15.4083 8.23161 13.2382 10.7799 11.7L13.1699 14.08Z" fill="black" />
                                    </g>
                                </svg>
                                </>    
                                :
                                <>
                                <svg width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                    <path d="M12 6.5C15.79 6.5 19.17 8.63 20.82 12C19.17 15.37 15.8 17.5 12 17.5C8.2 17.5 4.83 15.37 3.18 12C4.83 8.63 8.21 6.5 12 6.5ZM12 4.5C7 4.5 2.73 7.61 1 12C2.73 16.39 7 19.5 12 19.5C17 19.5 21.27 16.39 23 12C21.27 7.61 17 4.5 12 4.5ZM12 9.5C13.38 9.5 14.5 10.62 14.5 12C14.5 13.38 13.38 14.5 12 14.5C10.62 14.5 9.5 13.38 9.5 12C9.5 10.62 10.62 9.5 12 9.5ZM12 7.5C9.52 7.5 7.5 9.52 7.5 12C7.5 14.48 9.52 16.5 12 16.5C14.48 16.5 16.5 14.48 16.5 12C16.5 9.52 14.48 7.5 12 7.5Z" fill="#CED6DE" />
                                </svg>
                                </>
                                }
                            </span>
                        </div>
                        <div className="form-group mb-20 col-12 position-relative">
                            <InputField
                                register={register}
                                disabled={loading}
                                fieldClass="form-control shadow-none"
                                hideLabel={false}
                                labelClass="text-black-200 h6 font-weight-600 mb-2"
                                labelText="Confirm Password:"
                                placeHolder="Confirm Password"
                                errors={errors}
                                fieldType={showPassword1 ? "text" : "password"}
                                fieldId="passwordConfirmation"
                                fieldName="passwordConfirmation"
                                type={showPassword ? "text" : "password"}
                                aria-invalid={errors.name ? "true" : "false"}
                                autoComplete=""
                            />
                            <span className="show-password-trigger mt-3 me-2" onClick={handleClickShowPassword1}>
                                {showPassword1 ?
                                <>
                                <svg width="24" height="24" viewBox="0 0 36 36" fill="none" xmlns="http://www.w3.org/2000/svg">
                                    <g>
                                        <path d="M25.1901 20.3996C25.4773 19.6322 25.623 18.8191 25.6201 17.9996C25.6201 16.1803 24.8974 14.4354 23.6109 13.1489C22.3244 11.8624 20.5795 11.1396 18.7601 11.1396C17.9508 11.1406 17.1482 11.2862 16.3901 11.5696L18.0001 13.2296C18.2449 13.1905 18.4923 13.1704 18.7401 13.1696C20.0266 13.167 21.2618 13.6734 22.1761 14.5783C23.0905 15.4833 23.6096 16.7133 23.6201 17.9996C23.6193 18.2475 23.5993 18.4949 23.5601 18.7396L25.1901 20.3996Z" fill="black" />
                                        <path d="M34.29 17.5303C30.92 11.3003 25.01 7.53027 18.47 7.53027C16.6893 7.53445 14.9206 7.82136 13.23 8.38027L14.84 10.0003C16.0258 9.6938 17.2452 9.53593 18.47 9.53027C24.1 9.53027 29.22 12.6703 32.27 17.9603C31.1511 19.9229 29.6679 21.6539 27.9 23.0603L29.32 24.4803C31.3661 22.8296 33.0653 20.7905 34.32 18.4803L34.58 18.0003L34.29 17.5303Z" fill="black" />
                                        <path d="M4.86988 5.78004L9.32988 10.24C6.5108 12.0552 4.20678 14.5658 2.63988 17.53L2.37988 18L2.63988 18.48C6.00988 24.71 11.9199 28.48 18.4599 28.48C21.0125 28.4795 23.532 27.9018 25.8299 26.79L30.8299 31.79L32.5799 30.29L6.57988 4.29004L4.86988 5.78004ZM14.6199 15.53L21.2699 22.18C20.5185 22.6451 19.6535 22.8942 18.7699 22.9C18.1278 22.9001 17.4921 22.7731 16.8993 22.5265C16.3064 22.2799 15.7682 21.9185 15.3156 21.4631C14.863 21.0077 14.5049 20.4672 14.2619 19.8729C14.019 19.2786 13.8959 18.6421 13.8999 18C13.9112 17.1266 14.1601 16.2727 14.6199 15.53ZM13.1699 14.08C12.2343 15.3994 11.7958 17.0073 11.932 18.619C12.0682 20.2307 12.7703 21.7422 13.914 22.8859C15.0577 24.0296 16.5692 24.7317 18.1809 24.8679C19.7926 25.0041 21.4005 24.5656 22.7199 23.63L24.3199 25.23C22.4677 26.0218 20.4742 26.43 18.4599 26.43C12.8299 26.43 7.70988 23.29 4.65988 18C6.12359 15.4083 8.23161 13.2382 10.7799 11.7L13.1699 14.08Z" fill="black" />
                                    </g>
                                </svg>
                                </>    
                                :
                                <>
                                <svg width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                    <path d="M12 6.5C15.79 6.5 19.17 8.63 20.82 12C19.17 15.37 15.8 17.5 12 17.5C8.2 17.5 4.83 15.37 3.18 12C4.83 8.63 8.21 6.5 12 6.5ZM12 4.5C7 4.5 2.73 7.61 1 12C2.73 16.39 7 19.5 12 19.5C17 19.5 21.27 16.39 23 12C21.27 7.61 17 4.5 12 4.5ZM12 9.5C13.38 9.5 14.5 10.62 14.5 12C14.5 13.38 13.38 14.5 12 14.5C10.62 14.5 9.5 13.38 9.5 12C9.5 10.62 10.62 9.5 12 9.5ZM12 7.5C9.52 7.5 7.5 9.52 7.5 12C7.5 14.48 9.52 16.5 12 16.5C14.48 16.5 16.5 14.48 16.5 12C16.5 9.52 14.48 7.5 12 7.5Z" fill="#CED6DE" />
                                </svg>
                                </>
                                }
                            </span>
                        </div>
                        <div className="form-group col-12">
                            {!loading ? (
                                <button className="btn btn-primary w-100" type="submit">Submit</button>
                            ) : (
                                <div className="btn btn-primary w-100">Updating...</div>
                            )}
                        </div>
                    </form>
                </div>
            </div>
        </section>
    );
};

export default ForgotPasswordForm;
