'use client';
import { useState, useEffect, useRef } from "react";
import { useRouter } from "next/navigation";
import { useDispatch } from "react-redux";
import { useDropzone } from "react-dropzone";
import Image from "next/image";
import { Controller, useForm } from "react-hook-form";
import * as yup from 'yup';
import { yupResolver } from '@hookform/resolvers/yup';
import { DatePicker, SelectPicker } from 'rsuite';
import 'rsuite/dist/rsuite.min.css';
import moment from 'moment';
import InputField from "@/components/FormFields/InputField";
import TextAreaField from "../FormFields/TextAreaField";
import { getAuthToken } from "@/utils/auth";
import { fetchUserDetail, fetchCountries } from "@/utils/apiActions";
import { getUserDetailAction } from "@/store/slices/userSlice";

const schema = yup.object({
    fname: yup.string().required('First name is required.'),
    lname: yup.string().required('Last name is required.'),
});

const ProfileForm=()=>{
    const [submitLoading, setSubmitLoading] = useState(false);
    const [loading, setLoading] = useState(false);
    const [userData, setUserData] = useState({});
    const [isMounted, setIsMounted] = useState(false);
    const [errorMessage, setErrorMessage] = useState(null);
    const [successMessage, setSuccessMessage] = useState(null);
    const [fileError, setFileError] = useState(null);
    const [files ,setFiles] = useState([]);
    const [countries, setCountries] = useState([]);
    const dispatch = useDispatch();
    const router = useRouter();

    const token = getAuthToken();
    const { register, handleSubmit, control, setValue, formState: { errors } } = useForm({
        resolver: yupResolver(schema)
    });

    const selectPickerRef = useRef(null);
    const getUserDetail=async()=>{
        setLoading(true);
        const [response, responseCountries] = await Promise.all([
            fetchUserDetail(),
            fetchCountries()
        ]);
        setUserData(response.data);

        let countryOptions = [];
        if(responseCountries.data.length > 0){
            responseCountries.data.forEach(element => {
                countryOptions.push({value:element.id, label:element.name});                
            });
        }
        setCountries(countryOptions);
        setValue('fname',response.data.firstname);
        setValue('lname',response.data.lastname);
        if(response.data.dob!=='' && response.data.dob!==null){
            setValue('dob', new Date(response.data.dob));
        }
        if(response.data.country_id!=='' && response.data.country_id!==null){
            setValue('country',response.data.country_id);
        }
        setValue('address', response.data.address);
        setValue('phone', response.data.phone_number);
        setValue('bio', response.data.bio);
        setValue('postcode', response.data.postcode);
        setValue('state', response.data.state);
        if(response.data.gender!=='' && response.data.gender!==null){
            setValue('gender', response.data.gender);
        }
        setLoading(false);
        setValue('city', response.data.city);
        
    }

    const onSubmit = async (data) => {
        setErrorMessage(null);
        setSuccessMessage(null);
        setSubmitLoading(true);
        // Prepare form data
        const formData = new FormData();
        const secureHeader = new Headers();
        secureHeader.append("Accept", "application/json");
        secureHeader.append("Accept", "multipart/form-data");
        secureHeader.append("Authorization", `Bearer ${token}`);

        formData.append("firstname", data.fname || '');
        formData.append("lastname", data.lname || '');
        formData.append("phone_number", data.phone || '');
        formData.append("gender", data.gender || '');
        formData.append("address", data.address);
        formData.append("dob", moment(data.dob).format("YYYY-MM-DD") || '');
        formData.append("city",data.city || '');
        formData.append("bio", data.bio || '');
        formData.append("postcode", data.postcode || '');
        formData.append("country_id", data.country || '');
        formData.append("state", data.state || '');

        if(files.length>0){
            formData.append("photo", files[0].file);
        }
        const response = await fetch(`${process.env.NEXT_PUBLIC_API_URL}api/v1/user/update`, {
            method: "POST",
            headers: secureHeader,
            body: formData,
            redirect: "follow"
        });

        setSubmitLoading(false);

        const result = await response.json();
        if (result.status === 1) {
            dispatch(getUserDetailAction(result.data));
            setSuccessMessage(result.message);
            window.location.href= '/dashboard/profile';

        } else {
            setErrorMessage(result.message);
        }
    }

    const {isDragActive, getRootProps, getInputProps, open} = useDropzone({
        noClick: true,
        noKeyboard: true,
        accept: {
            'image/*': ['.png','.jpeg','.jpg','.gif','.webp'],             
        },
        maxFiles:1,
        //maxSize: 2 * 1024 * 1024,
        onDrop: acceptedFiles => {
            let tmpFiles = files;
            acceptedFiles.map((file) =>{
                let tmpObj={};
                tmpObj.file = file;
                tmpObj.preview = URL.createObjectURL(file);
                tmpObj.size = file.size;
                tmpObj.type = file.type;
                tmpObj.name = file.name;                
                tmpFiles.push(tmpObj);                
            });
            setFiles(tmpFiles);
        },
        onDropRejected:(rejections)=>{
            setFileError('File rejected : '+rejections);
        },
    });

    const disableFutureDates = (date) => {
        const today = new Date();
        return date > today; // Disable dates after today
    };

    useEffect(()=>{
        if(!isMounted){
            setIsMounted(true);
        }
        getUserDetail()
    },[isMounted]);
    return(
        <>
        {!loading &&
                <div className="profile-form-section">
                    <div className="container">
                        <div className="profile-form shadow rounded bg-white mx-auto p-3 p-sm-4">
                            <form onSubmit={handleSubmit(onSubmit)}>
                                <div className="row">
                                    <div className="form-group col-12 col-md-6 mb-20 text-center">
                                        {files.length<=0 ?
                                            <>
                                            <div {...getRootProps({className: `file-dropper file-dropper-circle mx-auto`})}>
                                                <input {...getInputProps()} />
                                                <div className={`file-dropper-inner position-relative`} onClick={open}>                                                    
                                                    <>
                                                        {Object.keys(userData).length>0 && userData?.photo!==null ?
                                                            <Image src={userData?.photo?.preview} 
                                                                alt="Uploaded Image" 
                                                                fill                                                                 
                                                                className="mb-2 rounded-circle"
                                                                style={{
                                                                    objectFit: 'cover',
                                                                }}
                                                                />
                                                        :
                                                            <svg className="mb-2" width="72" height="72" viewBox="0 0 72 72" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                                <g opacity="0.8">
                                                                    <path d="M57 15V57H15V15H57ZM57 9H15C11.7 9 9 11.7 9 15V57C9 60.3 11.7 63 15 63H57C60.3 63 63 60.3 63 57V15C63 11.7 60.3 9 57 9ZM42.42 35.58L33.42 47.19L27 39.42L18 51H54L42.42 35.58Z" fill="#FF3158" />
                                                                </g>
                                                            </svg>
                                                        }
                                                        </>                                                    
                                                    
                                                </div>
                                            </div>
                                            {fileError!==null && <p className='font-weight-bold d-block invalid-feedback'>{fileError}</p>}
                                            </>
                                            :
                                            <>
                                                <div className="file-dropper file-dropper-circle mb-20 mx-auto">
                                                    <div className="file-dropper-inner has-preview position-relative">
                                                        <Image 
                                                            src={files[0].preview} 
                                                            alt="Uploaded Preview" 
                                                            fill 
                                                            className="mb-2 rounded-circle"
                                                            style={{
                                                                objectFit: 'cover', // cover, contain, none
                                                            }}/>
                                                    </div>
                                                </div>
                                                <button type="button" className="btn btn-link p-0 text-decoration-none text-primary mx-auto" onClick={()=>{setFiles([])}}>Cancel</button>
                                            </>
                                        }
                                    </div>
                                    <div className="form-group mb-20 col-md-6">
                                        <div className="mb-20">
                                            <InputField
                                                    register={register}
                                                    disabled={loading}
                                                    fieldClass={'form-control shadow-none'}
                                                    hideLabel={false}
                                                    labelClass={'text-black-200 h6 font-weight-600 mb-2'}
                                                    labelText={'First Name:'}
                                                    placeHolder={'First Name'}
                                                    onFieldChange={(e)=>{}}
                                                    required={true}
                                                    errors={errors}
                                                    fieldType={"text"}
                                                    fieldId={"fname"}
                                                    fieldName={'fname'}
                                                    autoComplete={'fname'}
                                                />
                                        </div>
                                        <div>
                                            <InputField
                                                register={register}
                                                disabled={loading}
                                                fieldClass={'form-control shadow-none'}
                                                hideLabel={false}
                                                labelClass={'text-black-200 h6 font-weight-600 mb-2'}
                                                labelText={'Last Name:'}
                                                  placeHolder={'Last Name'}
                                                onFieldChange={(e)=>{}}
                                                required={true}
                                                errors={errors}
                                                fieldType={"text"}
                                                fieldId={"lname"}
                                                fieldName={'lname'}
                                                autoComplete={'lname'}
                                            />
                                        </div>
                                    </div>
                                    <div className="form-group mb-20 col-12 col-md-4">
                                        <InputField
                                            register={register}
                                            disabled={loading}
                                            fieldClass={'form-control shadow-none'}
                                            hideLabel={false}
                                            labelClass={'text-black-200 h6 font-weight-600 mb-2'}
                                            labelText={'Phone Number:'}
                                            placeHolder={'Phone Number'}
                                            onFieldChange={(e)=>{}}
                                            errors={errors}
                                            fieldType={"text"}
                                            fieldId={"phone"}
                                            fieldName={'phone'}
                                            autoComplete={'phone'}
                                        />
                                    </div>
                                    <div className="form-group mb-20 col-12 col-md-4">
                                        <label className="text-black-200 h6 font-weight-600 mb-2">Date Of Birth:</label>
                                        <Controller
                                            control={control}
                                            name='dob'
                                            errors={errors}
                                            render={({ field }) => (
                                                <DatePicker
                                                    {...field}
                                                    className="dtt-field"
                                                    value={field.value}
                                                    onChange={(value) => {
                                                        field.onChange(value);
                                                    }}
                                                    oneTap
                                                    placeholder="Select a date"
                                                    format="MM-dd-yyyy"
                                                    caretAs={null} 
                                                    cleanable={false}
                                                    shouldDisableDate={disableFutureDates}
                                                    block
                                                />
                                            )}
                                        />
                                        {errors && errors['startdate']?.message && (
                                            <p className='font-weight-bold d-block invalid-feedback'>{errors['startdate']?.message}</p>
                                        )}
                                    </div>
                                    <div className="form-group mb-20 col-12 col-md-4">
                                        <label className="text-black-200 h6 font-weight-600 mb-2 d-block">Select Gender:</label>
                                        <div className="d-flex w-100 pt-3">
                                            <div className="form-check form-check-inline">
                                                <input className="form-check-input" type="radio" id="inlinegender1" value="male" {...register('gender')} />
                                                <label className="form-check-label" htmlFor="inlinegender1">Male</label>
                                            </div>
                                            <div className="form-check form-check-inline">
                                                <input className="form-check-input" type="radio" id="inlinegender2" value="female" {...register('gender')}/>
                                                <label className="form-check-label" htmlFor="inlinegender2">Female</label>
                                            </div>
                                        </div>
                                        {errors && errors['gender']?.message && (
                                            <p className='font-weight-bold d-block invalid-feedback'>{errors['gender']?.message}</p>
                                        )}
                                    </div>
                                    <div className="form-group mb-20 col-12 col-md-6 col-lg-3">
                                        <InputField
                                            register={register}
                                            disabled={loading}
                                            fieldClass={'form-control shadow-none'}
                                            hideLabel={false}
                                            labelClass={'text-black-200 h6 font-weight-600 mb-2'}
                                            labelText={'City:'}
                                            placeHolder={'City'}
                                            onFieldChange={(e)=>{}}
                                            errors={errors}
                                            fieldType={"text"}
                                            fieldId={"city"}
                                            fieldName={'city'}
                                            autoComplete={'city'}
                                        />
                                    </div>
                                    <div className="form-group mb-20 col-12 col-md-6 col-lg-3">
                                        <InputField
                                            register={register}
                                            disabled={loading}
                                            fieldClass={'form-control shadow-none'}
                                            hideLabel={false}
                                            labelClass={'text-black-200 h6 font-weight-600 mb-2'}
                                            labelText={'State:'}
                                            placeHolder={'State'}
                                            onFieldChange={(e)=>{}}
                                            errors={errors}
                                            fieldType={"text"}
                                            fieldId={"state"}
                                            fieldName={'state'}
                                            autoComplete={'state'}
                                        />
                                    </div>
                                    <div className="form-group mb-20 col-12 col-md-6 col-lg-3">
                                        <InputField
                                            register={register}
                                            disabled={loading}
                                            fieldClass={'form-control shadow-none'}
                                            hideLabel={false}
                                            labelClass={'text-black-200 h6 font-weight-600 mb-2'}
                                            labelText={'Postcode:'}
                                            placeHolder={'Postcode'}
                                            onFieldChange={(e)=>{}}
                                            errors={errors}
                                            fieldType={"text"}
                                            fieldId={"postcode"}
                                            fieldName={'postcode'}
                                            autoComplete={'postcode'}
                                        />
                                    </div>
                                    <div className="form-group mb-20 col-12 col-md-6 col-lg-3">
                                        <label className="text-black-200 h6 font-weight-600 mb-2">Country:</label>
                                        <Controller
                                            control={control}
                                            name="country"
                                            errors={errors}
                                            render={({ field }) => {
                                                return (
                                                    <SelectPicker
                                                        ref={selectPickerRef} // Attach ref to access methods
                                                        className="dtt-field"
                                                        data={countries}
                                                        value={field.value}
                                                        block
                                                        placeholder="Select country"
                                                        onFocus={() => {
                                                            if (selectPickerRef.current) {
                                                                selectPickerRef.current.open(); // Open dropdown on focus
                                                            }
                                                        }}
                                                        onChange={(value) => field.onChange(value)} // Handle value change
                                                    />
                                                );
                                            }}
                                        />
                                        {errors?.country?.message && (
                                            <p className="font-weight-bold d-block invalid-feedback">
                                                {errors['country']?.message}
                                            </p>
                                        )}
                                    </div>
                                    <div className="form-group mb-20 col-12 col-lg-6">
                                        <TextAreaField
                                            register={register}
                                            disabled={loading}
                                            fieldClass={'form-control shadow-none'}
                                            hideLabel={false}
                                            labelClass={'text-black-200 h6 font-weight-600 mb-2'}
                                            labelText={'Adress:'}
                                            placeHolder={'Adress'}
                                            onFieldChange={(e)=>{}}
                                            errors={errors}
                                            fieldType={"text"}
                                            fieldId={"address"}
                                            fieldName={'address'}
                                            autoComplete={""}
                                        />
                                    </div>
                                    <div className="form-group mb-20 col-12 col-lg-6">
                                        <TextAreaField
                                            register={register}
                                            disabled={loading}
                                            fieldClass={'form-control shadow-none'}
                                            hideLabel={false}
                                            labelClass={'text-black-200 h6 font-weight-600 mb-2'}
                                            labelText={'Bio:'}
                                            placeHolder={'Bio'}
                                            onFieldChange={(e)=>{}}
                                            errors={errors}
                                            fieldType={"text"}
                                            fieldId={"bio"}
                                            fieldName={'bio'}
                                            autoComplete={""}
                                        />
                                    </div>
                                </div>
                                <div className="form-group col-12">
                                {!submitLoading ?
                                    <button className="btn btn-primary" type="submit">Update Profile</button>
                                    :
                                    <div className="btn btn-primary">Updating...</div>
                                }
                                </div>
                            </form>
                            {errorMessage!==null &&
                                <div className="alert alert-danger mt-3" role="alert">{errorMessage}</div>
                            }
                            {successMessage!==null &&
                                <div className="alert alert-success mt-3" role="alert">{successMessage}</div>
                            }
                        </div>
                    </div>
                </div>
            }
        </>
    )
}

export default ProfileForm;