'use client';
import { useState, useEffect, useRef } from "react";
import { useForm } from "react-hook-form";
import * as yup from 'yup';
import { yupResolver } from '@hookform/resolvers/yup';
import { fetchByEmail } from "@/utils/apiActions";
import Image from "next/image";
import 'rsuite/dist/rsuite.min.css';
import { getAuthToken } from "@/utils/auth";
const schema = yup.object({
    searchQuery: yup
        .string()
        .required('Please enter a name or email to search.')
        .test('is-valid-query', 'Please enter a valid input.', (value) => {
            return !!value.trim();
        }),
}).required();

const FoundMemberRow=(props)=>{
    const {member, token} = props;
    
    const [successMessage, setSuccessMessage] = useState(null);
    const [loading, setLoading]= useState(false);
    const [errorMessage2, setErrorMessage2]= useState(null);

    const onSubmitRequest = async () => {
        setLoading(true);
        setSuccessMessage(null);
        
        const secureHeader = new Headers();
        secureHeader.append("Accept", "application/json");
        secureHeader.append("Accept", "multipart/form-data");
        secureHeader.append("Authorization", `Bearer ${token}`);

        const formdata = new FormData();
        formdata.append("connection_id", member.user.id);

        const requestOptions = {
            method: "POST",
            headers: secureHeader,
            body: formdata,
            redirect: "follow"
        };

        try {
            const response = await fetch(`${process.env.NEXT_PUBLIC_API_URL}api/v1/connections`, requestOptions);
            const result = await response.text();
            let rdata = JSON.parse(result);
            setLoading(false);
            setSuccessMessage("Request Sent Successfully.");
        } catch (error) {
            console.error("Error: ", error);
            setLoading(false);
            setSuccessMessage("An error occurred. Please try again.");
        }
    };

    const [buttonClicked, setButtonClicked] = useState(false);

    const inviteButtonClick = () => {
        setButtonClicked(true);
        onSubmitRequest();
        setTimeout(() => {
            window.location.href = '/dashboard/friends-family/my-connections/invitations';
        }, 3000);
    };


    return(
        <div className={`my-connection-card d-flex gap-2 gap-md-4 p-2 p-md-3 shadow-sm rounded mb-4`}>
            <div className={`position-relative rounded-circle user-pic-50`}>                                                    
                {member.user.photo ?
                    <Image 
                        src={member.user.photo.preview} 
                        alt="" 
                        fill 
                        className="mb-2 rounded-circle"
                        style={{
                            objectFit: 'cover',
                        }}
                    />
                    :
                    <i className="ti-user"></i>
                }
            </div>
            <div className="flex-grow-1 d-flex flex-column justify-content-between mb-2">
                <div className="usermeta">
                    <div className="user-name text-primary">{member.user.firstname} {member.user.lastname}</div>
                    <div className="user-location">
                    {member.user.state && member.user.country_name 
                        ? `${member.user.state}, ${member.user.country_name}` 
                        : member.user.state || member.user.country_name}
                    </div>
                </div>
                {member?.connection_status === 'Connected' ? (
                    <p className="connection-status text-success">{member.connection_status}</p>
                ) : member?.connection_status === 'Requestsent' ? (
                    <p className="connection-status text-success">{member.connection_status}</p>
                ) : (
                    <button type="button" onClick={inviteButtonClick} className="invite-btn">
                        {loading ? "Sending..." : "Invite"}
                    </button>
                )}

            </div>
        </div>
    )
}

const SendInvitation = (props) => {
    const { setShowPopup } = props;
    const [loading, setLoading] = useState(false);
    const [isMounted, setIsMounted] = useState(false);
    const [memberItem, setMemberItem] = useState([]);
    const [errorMessage1, setErrorMessage1] = useState(null);
    const [successMessage1, setSuccessMessage1] = useState(null);

    const { register, handleSubmit, formState: { errors } } = useForm({
        resolver: yupResolver(schema)
    });

    const token = getAuthToken();

    const closePopup = () => {
        setShowPopup(false);
    };
    const [searchQuery, setSearchQuery] = useState("");

    const onSubmit = async (data) => {
        setLoading(true);
        setErrorMessage1(null);
        setMemberItem([]);
        setSearchQuery(data.searchQuery);
        const [membersData] = await Promise.all([fetchByEmail(data.searchQuery)]);
        setMemberItem(membersData.data);
        setSuccessMessage1('Member Found!!');
        setLoading(false);
    };
    

    useEffect(() => {
        if (!isMounted) {
            setIsMounted(true);
            return;
        }
    }, [isMounted]);

    const [message, setMessage] = useState("");
    const [showMessage, setShowMessage] = useState(true);
    const handleInviteClick = async (email) => {
        setShowMessage(false);
        const secureHeader = new Headers();
        secureHeader.append("Authorization", `Bearer ${token}`);
        const formdata = new FormData();
        formdata.append("email", email);
    
        const requestOptions = {
            method: "POST",
            headers: secureHeader,
            body: formdata, 
            redirect: "follow"
        };
    
        await fetch(`${process.env.NEXT_PUBLIC_API_URL}api/v1/invite`, requestOptions)
        .then((response) => response.text())
        .then((result) => {
            const apiMessage = result.message || "Invitation sent successfully!";
            setMessage(`${apiMessage}`);
            setTimeout(() => {
                window.location.reload();
            }, 2000);
        })
        .catch((error) => {
            console.error(error);
            const apiMessage = result.message || "Failed to send invite. Please try again.";
            setMessage(`${apiMessage}`);
        })
        .finally(() => {
            setLoading(false);
        });
    };
    
    

    return (
        <>
            <div className="popup-screen">
                <div className="popup-screen-inner position-relative send-invitation-popup">
                    <button type="button" className="popup-screen-close" onClick={closePopup}>
                        <i className="ti-close"></i>
                    </button>
                    <div className="px-4 text-center mb-4">
                        <h4>Send <span className="text-primary">Invitation</span></h4>
                    </div>
                    <div className="popup-screen-content">
                        <form onSubmit={handleSubmit(onSubmit)} className="row justify-content-center mb-4">
                            <div className="col-12">
                                <div className="d-flex gap-2 items-center">
                                    <input
                                        type="text"
                                        className="form-control shadow-none"
                                        placeholder="Search Member by Email"
                                        {...register('searchQuery')}
                                    />
                                    <button type="submit" className="btn btn-primary" disabled={loading}>
                                        Find
                                    </button>
                                </div>
                                {errors && errors['searchQuery']?.message && (
                                    <p className="font-weight-bold d-block invalid-feedback">
                                        {errors['searchQuery']?.message}
                                    </p>
                                )}
                                {errorMessage1 !== null && (
                                    <p className="alert alert-danger mt-3 p-2 fs-12 text-center">
                                        {errorMessage1}
                                    </p>
                                )}
                            </div>
                        </form>

                        {loading && (
                            <div className="css-loader w-100">
                                <div className="lds-ripple">
                                    <div></div>
                                    <div></div>
                                </div>
                            </div>
                        )}

                        {memberItem?.length > 0 ? (
                            <>
                                <label className="fw-bold mb-1">Found Member:</label>
                                <div className="popup-screen-wrap">
                                    {memberItem.map((item, index) => (
                                        <FoundMemberRow member={item} key={index} token={token} />
                                    ))}
                                </div>
                            </>
                        ) : memberItem?.length === 0 ? (
                            <p className="alert alert-warning mt-3 p-2 fs-14 text-center">
                                No members found for your search query.
                            </p>
                        ) : (
                            <div className="text-center">
                                {showMessage && (
                                    <>
                                        <p className="alert alert-info mt-3 p-2 fs-14 text-center">
                                            Oops !! This user is still not registered with us on During Those Times platform. Would you like to invite?
                                        </p>
                                        <button type="button" className="btn btn-primary btn-sm" onClick={() => handleInviteClick(searchQuery)}>
                                            {loading ? "Inviting..." : "Invite Now"}
                                        </button>
                                    </>
                                )}
                                {message && (
                                    <p className="alert alert-success mt-3 p-2 fs-14">{message}</p>
                                )}
                            </div>
                        )}
                    </div>
                </div>
            </div>
        </>
    );
};

export default SendInvitation;
