"use client";

import { useState } from "react";
import { useRouter } from "next/navigation";
import { useForm } from "react-hook-form";
import { useDispatch } from "react-redux";
import * as yup from 'yup';
import { loginAction } from "@/store/slices/userSlice";
import InputField from "@/components/FormFields/InputField";
import PasswordField from "@/components/FormFields/PasswordField";
import { yupResolver } from '@hookform/resolvers/yup';
import Link from "next/link";

const schema = yup.object({
    username: yup.string().email('Invalid Email').required('Please Enter your email').matches(/^(([^<>()\[\]\\.,;:\s@"]+(\.[^<>()\[\]\\.,;:\s@"]+)*)|(".+"))@((\[[0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}])|(([a-zA-Z\-0-9]+\.)+[a-zA-Z]{2,}))$/, "Invalid Email"),
    password: yup.string().required('Please Enter your password'),    
}).required();

const LoginForm=()=>{
    const [loading, setLoading] = useState(false);
    const [errorMessage, setErrorMessage] = useState(null);
    const router = useRouter();
    const { register, handleSubmit, formState: { errors } } = useForm({
        resolver: yupResolver(schema)
    });
    const dispatch = useDispatch();
    const onSubmit = async (data) => {
        setErrorMessage(null);
        setLoading(true);
    
        // Prepare form data
        const formData = new FormData();
        formData.append("email", data.username);
        formData.append("password", data.password);
    
        // Make the fetch request
        const response = await fetch(`${process.env.NEXT_PUBLIC_API_URL}api/login`, {
            method: "POST",
            body: formData,
        });    
        setLoading(false);
        const result = await response.json();        
        if (result.status === 1) {
            dispatch(loginAction(result));
            router.push('/dashboard');
        } else {
            setErrorMessage(result.message);
        }
    };
    return(
        <>
        <section className="login-form-section">
            <div className="container">
                <div className="login-form shadow rounded bg-white mx-auto p-3 p-sm-4">
                    {errorMessage!==null &&
                        <div className="alert alert-danger" role="alert">{errorMessage}</div>
                    }
                    <form onSubmit={handleSubmit(onSubmit)} className={`row`}>
                        <div className="form-group mb-20 col-12">
                            <InputField
                                register={register}
                                disabled={loading}
                                fieldClass={'form-control shadow-none'}
                                hideLabel={false}
                                labelClass={'text-black-200 h6 font-weight-600 mb-2'}
                                labelText={'Email Address:'}
                                placeHolder={'Email Address'}
                                onFieldChange={(e)=>{}}
                                required={true}
                                errors={errors}
                                fieldType={"text"}
                                fieldId={"username"}
                                fieldName={'username'}
                                autoComplete={"username"}
                            />
                        </div>
                        <div className="form-group mb-20 col-12">
                            <PasswordField
                                register={register}
                                disabled={loading}
                                fieldClass={'form-control shadow-none'}
                                fieldName={'password'}
                                labelClass={'text-black-200 h6 font-weight-600 mb-2'}
                                fieldId={'passwordSignIn'}
                                hideLabel={false}
                                labelText={'Password:'}
                                placeHolder={'Password'}
                                required={true}
                                errors={errors}
                                autoComplete={"current-password"}
                                visibleTrigger={true}
                            />                            
                        </div>
                        <div className="mb-3">
                        </div>
                        <div className="form-group col-12">
                        {!loading ?
                            <button className="btn btn-primary w-100" type="submit">Login</button>
                            :
                            <div className="btn btn-primary w-100">Updating...</div>
                        }
                        </div>
                        <div className="text-center mt-3">
                            <Link href="/auth/forgot-password" className="text-decoration-underline text-primary">Forgot Password?</Link>
                        </div>
                    </form>
                </div>
            </div>
        </section>
        </>
    )
}

export default LoginForm;