"use client";
import { useState, useEffect } from "react";
import { useRouter } from "next/navigation";
import { useDropzone } from "react-dropzone";
import Image from "next/image";
import * as yup from 'yup';
import { yupResolver } from '@hookform/resolvers/yup';
import 'rsuite/dist/rsuite.min.css';
import InputField from "@/components/FormFields/InputField";
import { fetchKeepSakesSingle } from "@/utils/apiActions";
import { getAuthToken } from "@/utils/auth";
import { useForm } from "react-hook-form";

const schema = yup.object({
    title: yup.string().required('Keepsakes title is required.')
}).required();

const KeepsakesForm=(props)=>{
    const {isEdit, keepsakesId} = props;
    const [keepsakers, setKeepsakers] = useState({});          
    const [loading, setLoading] = useState(false);
    const [submitLoading, setSubmitLoading] = useState(false);
    const [errorMessage, setErrorMessage] = useState(null);
    const [fileError, setFileError] = useState(null);
    const [files ,setFiles] = useState([]);
    const router = useRouter();
    const [isMounted, setIsMounted] = useState(false);

    const { register, handleSubmit, setValue, formState: { errors } } = useForm({
        resolver: yupResolver(schema)
    });
    const token=getAuthToken();   

    const fetchKeepsakeData = async () => {
        if (isEdit && keepsakesId) {
            setLoading(true);
            try {
                const data = await fetchKeepSakesSingle(keepsakesId, token);
                setKeepsakers(data);
                setValue("title", data.title);
                setLoading(false);
            } catch (error) {
                setLoading(false);
                setErrorMessage("Failed to fetch keepsake details.");
            }
        }
    };
    useEffect(() => {
        fetchKeepsakeData();
    }, [isEdit, keepsakesId]);

    const addChallangeSubmit = async(data)=>{
        setErrorMessage(null);
        const secureHeader = new Headers();
        secureHeader.append("Accept", "application/json");
        secureHeader.append("Accept", "multipart/form-data");
        secureHeader.append("Authorization", `Bearer ${token}`);

        const formdata = new FormData();
        formdata.append("title", data.title);
        
        if(files.length>0){
            formdata.append("photo", files[0].file);
        }

        const requestOptions = {
            method: "POST",
            headers: secureHeader,
            body: formdata,
            redirect: "follow"
        };
        
        setSubmitLoading(true);

        await fetch(`${process.env.NEXT_PUBLIC_API_URL}api/v1/keeps-sakes`, requestOptions)
        .then((response) => response.text())
        .then((result) => {
            let rdata = JSON.parse(result);
            setSubmitLoading(false);    
            router.push('/dashboard/friends-family/keepsakes');
        })
        .catch((error) => {
            setSubmitLoading(false);
            setErrorMessage(error);
        });
    }
    const onSubmit = async (data) => {
        addChallangeSubmit(data);
    };
    

    const { isDragActive, getRootProps, getInputProps, open } = useDropzone({
        noClick: true,
        noKeyboard: true,
        accept: {
            'image/*': ['.png', '.jpeg', '.jpg', '.gif', '.webp'],
            'audio/*': ['.mp3', '.wav'],
            'video/*': ['.mp4', '.avi', '.mov', '.mkv'],
            'application/pdf': ['.pdf'],
            'application/msword': ['.doc'],
            'application/vnd.openxmlformats-officedocument.wordprocessingml.document': ['.docx']
        },
        maxFiles: 1,
        onDrop: acceptedFiles => {
            let tmpFiles = [...files];
            acceptedFiles.forEach((file) => {
                let tmpObj = {
                    file,
                    preview: URL.createObjectURL(file),
                    size: file.size,
                    type: file.type,
                    name: file.name
                };
                tmpFiles.push(tmpObj);
            });
            setFiles(tmpFiles);
        },
        onDropRejected: (rejections) => {
            setFileError('File rejected: ' + rejections.map(r => r.file.name).join(', '));
        }
    });
    
    

    useEffect(()=>{
        if (!isMounted) {
            setIsMounted(true);
            return;
        }   
    },[isMounted]);

    return(
        <>
            {!loading &&
                <div className="event-form-section">
                    <div className="container">
                        <div className="event-form shadow rounded bg-white mx-auto p-3 p-sm-4">
                            <form onSubmit={handleSubmit(onSubmit)}>
                                <div className="row">
                                    <div className="form-group mb-20 col-12">
                                        <InputField
                                            register={register}
                                            disabled={loading}
                                            fieldClass={'form-control shadow-none'}
                                            hideLabel={false}
                                            labelClass={'text-black-200 h6 font-weight-600 mb-2'}
                                            labelText={'Title:'}
                                            placeHolder={'Title'}
                                            onFieldChange={(e)=>{}}
                                            required={true}
                                            errors={errors}
                                            fieldType={"text"}
                                            fieldId={"title"}
                                            fieldName={'title'}
                                            autoComplete={""}
                                        />
                                    </div>
                                    <div className="form-group mb-20 col-12 col-md-6">
                                        <label className="text-black-200 h6 font-weight-600 mb-2">File:</label>
                                        {files.length <= 0 ? (
                                        <div {...getRootProps({ className: `file-dropper` })}>
                                            <input {...getInputProps()} />
                                            <div className={`file-dropper-inner`} onClick={open}>
                                                <>
                                                    {/* {Object.keys(keepsakers).length > 0 &&
                                                    isEdit !== undefined &&
                                                    isEdit &&
                                                    keepsakers.photo &&
                                                    keepsakers.photo.original_url ? (
                                                        <Image 
                                                            src={keepsakers.photo.original_url} 
                                                            alt="Uploaded Image" 
                                                            width={200} 
                                                            height={200} 
                                                            className="mb-2" 
                                                            style={{ objectFit: 'cover' }} 
                                                        />
                                                    ) : (
                                                    )} */}
                                                        <svg className="mb-3" xmlns="http://www.w3.org/2000/svg" version="1.1" xmlnsXlink="http://www.w3.org/1999/xlink" width="80" height="80" viewBox="0 0 791.454 791.454" style={{ enableBackground: 'new 0 0 512 512' }}>
                                                        <g>
                                                          <path fill="#ff3158" d="M202.621 0h264.884l224.238 233.828v454.717c0 56.783-46.126 102.909-103.061 102.909h-386.06c-56.783 0-102.909-46.126-102.909-102.909V102.909C99.712 46.126 145.838 0 202.621 0z" />
                                                          <g fill="#fff">
                                                            <path fillRule="evenodd" d="M467.2 0v232.002h224.542z" clipRule="evenodd" opacity="1" fill="#ffffff302" />
                                                            <path d="M500.082 411.788h-208.71c-9.591 0-17.507-7.764-17.507-17.354s7.916-17.354 17.507-17.354h208.71c9.591 0 17.354 7.764 17.354 17.354s-7.763 17.354-17.354 17.354zm-69.57 208.71h-139.14c-9.591 0-17.507-7.764-17.507-17.354s7.916-17.354 17.507-17.354h139.14c9.591 0 17.354 7.764 17.354 17.354s-7.763 17.354-17.354 17.354zm69.57-69.57h-208.71c-9.591 0-17.507-7.764-17.507-17.354s7.916-17.354 17.507-17.354h208.71c9.591 0 17.354 7.764 17.354 17.354s-7.763 17.354-17.354 17.354zm0-69.57h-208.71c-9.591 0-17.507-7.764-17.507-17.354s7.916-17.354 17.507-17.354h208.71c9.591 0 17.354 7.764 17.354 17.354s-7.763 17.354-17.354 17.354z" fill="#ffffff" />
                                                          </g>
                                                        </g>
                                                      </svg>

                                                    {isDragActive ? (
                                                        <p>Drop the file here...</p>
                                                    ) : (
                                                        <>
                                                            <p>Drag & Drop Or <span type="button" className="text-primary fw-bold">Choose file</span> to upload</p>
                                                            <span>(JPG, JPEG, GIF, WEBP, PNG, PDF, DOC, DOCX, MP3, WAV, MP4, AVI, MOV, MKV  )</span>
                                                        </>

                                                    )}
                                                    {fileError !== null && <p className="font-weight-bold d-block invalid-feedback">{fileError}</p>}
                                                </>
                                            </div>
                                        </div>
                                    ) : (
                                        <div className="file-dropper">
                                            <div className="file-dropper-inner has-preview position-relative">
                                                {files[0].type === "application/pdf" ? (
                                                    <div>
                                                        <p>PDF File: {files[0].name}</p>
                                                        <button 
                                                            type="button" 
                                                            className="btn btn-primary mx-auto mb-2" 
                                                            onClick={() => window.open(files[0].preview, "_blank")}
                                                        >
                                                            Open PDF
                                                        </button>
                                                        <button 
                                                            type="button" 
                                                            className="btn btn-danger mx-auto" 
                                                            onClick={() => setFiles([])}
                                                        >
                                                            Remove
                                                        </button>
                                                    </div>
                                                ) : (
                                                    <>
                                                         <svg className="mb-3" xmlns="http://www.w3.org/2000/svg" version="1.1" xmlnsXlink="http://www.w3.org/1999/xlink" width="80" height="80" viewBox="0 0 791.454 791.454" style={{ enableBackground: 'new 0 0 512 512' }}>
                                                            <g>
                                                            <path fill="#ff3158" d="M202.621 0h264.884l224.238 233.828v454.717c0 56.783-46.126 102.909-103.061 102.909h-386.06c-56.783 0-102.909-46.126-102.909-102.909V102.909C99.712 46.126 145.838 0 202.621 0z" />
                                                            <g fill="#fff">
                                                                <path fillRule="evenodd" d="M467.2 0v232.002h224.542z" clipRule="evenodd" opacity="1" fill="#ffffff302" />
                                                                <path d="M500.082 411.788h-208.71c-9.591 0-17.507-7.764-17.507-17.354s7.916-17.354 17.507-17.354h208.71c9.591 0 17.354 7.764 17.354 17.354s-7.763 17.354-17.354 17.354zm-69.57 208.71h-139.14c-9.591 0-17.507-7.764-17.507-17.354s7.916-17.354 17.507-17.354h139.14c9.591 0 17.354 7.764 17.354 17.354s-7.763 17.354-17.354 17.354zm69.57-69.57h-208.71c-9.591 0-17.507-7.764-17.507-17.354s7.916-17.354 17.507-17.354h208.71c9.591 0 17.354 7.764 17.354 17.354s-7.763 17.354-17.354 17.354zm0-69.57h-208.71c-9.591 0-17.507-7.764-17.507-17.354s7.916-17.354 17.507-17.354h208.71c9.591 0 17.354 7.764 17.354 17.354s-7.763 17.354-17.354 17.354z" fill="#ffffff" />
                                                            </g>
                                                            </g>
                                                        </svg>
                                                        <button 
                                                            type="button" 
                                                            className="btn btn-primary mx-auto" 
                                                            onClick={() => setFiles([])}
                                                        >
                                                            Cancel
                                                        </button>
                                                    </>
                                                )}
                                            </div>
                                        </div>
                                    )}

                                    </div>
                                </div>
                                <div className="form-group col-12">
                                {!submitLoading ?
                                    <button className="btn btn-primary" type="submit">{isEdit!==undefined && isEdit ? 'Update Keepsakes' : 'Create Keepsakes'}</button>
                                    :
                                    <div className="btn btn-primary">Updating...</div>
                                }
                                </div>
                            </form>
                            {errorMessage!==null &&
                                <div className="alert alert-danger mt-3" role="alert">{errorMessage}</div>
                            }
                        </div>
                    </div>
                </div>
            }
        </>
    )
}

export default KeepsakesForm;