"use client";

import { useState, useEffect, useRef } from 'react';
import { usePathname } from 'next/navigation';
import { useSelector, useDispatch } from 'react-redux';
import { isUserAuthenticated, logoutUser } from '@/utils/auth';
import Image from "next/image";
import Link from "next/link";
import Logo from "../../../public/images/logo.png";
import { fetchUserDetail } from '@/utils/apiActions';
import { getUserDetailAction } from '@/store/slices/userSlice';

const DashboardHeader=()=>{
    const [isLoggedin, setIsLoggedin] = useState(false);
    const [loading, setLoading] = useState(false);
    const [showUserNavModel, setShowUserNavModel] = useState(false);
    const [isMounted, setIsMounted] = useState(false);
    const [isMenuOpen, setIsMenuOpen] = useState(false);
    const [isScrolled, setIsScrolled] = useState(false);
    
    const userObj = useSelector(state=>state.user);
    const currentUser=userObj.user;
    const dispatch=useDispatch();

    const handleLogout = () => {
        logoutUser();
        // Optionally redirect or refresh the page after logout
        window.location.href = '/'; // or any other route
        if (isMenuOpen) {
            toggleMobileMenu(); // Close the menu when a link is clicked
        }
    };
    const pathname = usePathname();
    const refMenu=useRef();
    
    const navItems = [
        { name: 'Welcome', href: '/dashboard' },
        { name: 'Events', href: '/dashboard/events' },
        { name: 'Friends & Family', href: '/dashboard/friends-family' },
        { name: 'Wellness Center', href: '/dashboard/wellness-center' },
        { name: 'Resource Center', href: '/dashboard/resource-center' },
    ];

    const isActive = (href) => {
        if (href === '/dashboard') {
            return pathname === href; // Only match exact path for Dashboard
        }
        
        return pathname === href || pathname.startsWith(href);
    };

    
    const handleUserNavModel=()=>{
        setShowUserNavModel(!showUserNavModel);
    }
    const toggleMobileMenu = () => {
        setIsMenuOpen((prevState) => {
            const newState = !prevState;
            if (newState) {
                document.body.classList.add('menu-open'); // Add class to body
            } else {
                document.body.classList.remove('menu-open'); // Remove class from body
            }
            return newState;
        });
    };
    const getUserDetail=async()=>{
        setLoading(true);
        const [response] = await Promise.all([
            fetchUserDetail()
        ]);
        dispatch(getUserDetailAction(response.data));
        setLoading(false);
    }

    useEffect(() => {
        // Check auth status on the client
        setIsLoggedin(isUserAuthenticated());
    }, []);

    useEffect(()=>{
        const checkIfClickedOutside = e => {
            // If the menu is open and the clicked target is not within the menu,
            // then close the menu
            if ((showUserNavModel) && refMenu.current && !refMenu.current.contains(e.target)) {
                setShowUserNavModel(false);
            }
        }      
        document.addEventListener("mousedown", checkIfClickedOutside);      
        return () => {
            document.removeEventListener("mousedown", checkIfClickedOutside)
        }
    },[showUserNavModel]);

    useEffect(()=>{
        if(!isMounted){
            setIsMounted(true);
            return ;
        }
        getUserDetail();
    },[isMounted]);
    const closeMenuOnLinkClick = () => {
        if (isMenuOpen) {
            toggleMobileMenu(); // Close the menu when a link is clicked
        }
        setShowUserNavModel(false);
    };
    
    useEffect(() => {
        const handleScroll = () => {
            if (window.scrollY > 70) { // Adjust the scroll position threshold as needed
                setIsScrolled(true);
            } else {
                setIsScrolled(false);
            }
        };
        window.addEventListener("scroll", handleScroll);
        
        // Clean up the event listener
        return () => {
            window.removeEventListener("scroll", handleScroll);
        };
    }, []);
    return(
    <>
  
        <header className={`header-nav position-relative bg-light-gray has-shapes has-bg-brash bg-brash-bottom ${
                isScrolled ? "scrolled" : ""
            }`}>
            <div className="container">
                <nav className="navbar navbar-expand-xl navbar-light px-0">
                    {isLoggedin ?
                    <Link className="navbar-brand p-0" href="/dashboard">
                        <Image src={Logo} width={80} height={84} alt="During Those Time" priority />
                    </Link>
                    :
                    <Link className="navbar-brand p-0" href="/">
                        <Image src={Logo} width={80} height={84} alt="During Those Time" priority />
                    </Link>
                    }

                        <button
                            className={`navbar-toggler bg-white rounded-0 p-0  ${
                                isMenuOpen ? "active" : ""
                            }`}
                            type="button"
                            onClick={toggleMobileMenu} // Use React state to toggle menu
                        >
                            <svg className="nav-toggle-icon" viewBox="0 0 100 100" width="40">
                                <path className="line top" d="m 70,33 h -40 c 0,0 -8.5,-0.149796 -8.5,8.5 0,8.649796 8.5,8.5 8.5,8.5 h 20 v -20" />
                                <path className="line middle" d="m 70,50 h -40" />
                                <path className="line bottom" d="m 30,67 h 40 c 0,0 8.5,0.149796 8.5,-8.5 0,-8.649796 -8.5,-8.5 -8.5,-8.5 h -20 v 20" />
                            </svg>
                        </button>
                    
                    <div className={`collapse navbar-collapse ${isMenuOpen ? 'show' : ''}`} id="navlinks">
                        <ul className="navbar-nav mx-auto">
                                {currentUser.subscription_status !== false &&
                                <>
                                    {navItems.map((item,index) => (
                                        <li key={index} className={`nav-item ${isActive(item.href) ? 'active' : ''}`}>
                                            <Link className="nav-link" href={item.href} onClick={closeMenuOnLinkClick}>{item.name}</Link>
                                        </li>
                                    ))}
                                </>
                                }
                               <div className='d-block d-lg-none col-12'>
                                    <div className='d-flex nav-item'>
                                        <Link href={`/dashboard/profile`} type='button'  onClick={closeMenuOnLinkClick} className='nav-link btn-outline-primary text-decoration-none '>Profile</Link>
                                    </div>
                                    <div className='d-flex flex-column align-items-start nav-item'>
                                        {/* <Link href={`/dashboard/profile/edit`} onClick={closeMenuOnLinkClick} className="nav-link py-0 px-0 text-decoration-none btn-link ">Edit Profile</Link> */}
                                        <Link href={`/dashboard/profile/change-password`} onClick={closeMenuOnLinkClick} className="nav-link py-0 px-0 text-decoration-none btn-link ">Change Password</Link>
                                        <button onClick={handleLogout}  className="nav-link py-0 px-0 text-decoration-none btn-link">Logout</button>
                                    </div>
                                </div> 
                        </ul>
                        {(!loading && Object.keys(currentUser).length>0) &&
                            <>
                                <div className="navbar-button d-none d-lg-block position-relative" ref={refMenu}>
                                    <button type="button" onClick={handleUserNavModel} className="btn btn-sm btn-outline-primary d-flex align-items-center">Hi, {`${currentUser.firstname} ${currentUser.lastname}`}<i className='ti-angle-down ms-2'></i></button>
                                    <div className={`nav-user-modal ${showUserNavModel && 'active'}`}>
                                        <div className='border-bottom border-gray pb-3 mb-3'>
                                            <div className='d-flex align-items-center mb-3'>
                                                <div className="profile-image overflow-hidden rounded-circle">
                                                    {currentUser.photo!==null ?
                                                    <>
                                                        <Image src={currentUser.photo.thumbnail} width={50} height={50} alt="During Those Time" priority />
                                                    </>
                                                    :
                                                    <>
                                                        <i className="ti-user"></i>
                                                    </>
                                                    }
                                                </div>
                                                <div className='fw-bold flex-grow-1 ps-3'>{`${currentUser.firstname} ${currentUser.lastname}`}</div>
                                            </div>
                                            <div className='d-flex justify-content-center'>
                                                <Link href={`/dashboard`} onClick={closeMenuOnLinkClick} type='button' className='btn btn-tiny mx-auto btn-outline-primary text-decoration-nones text-uppercase'>Dashboard</Link>
                                            </div>
                                        </div>
                                        <div className='d-flex flex-column align-items-start'>
                                            <Link href={`/dashboard/profile/edit`} onClick={closeMenuOnLinkClick} className="btn py-0 px-0 text-decoration-none btn-link mb-2">Edit Profile</Link>
                                            <Link href={`/dashboard/profile/change-password`} onClick={closeMenuOnLinkClick} className="btn py-0 px-0 text-decoration-none btn-link mb-2">Change Password</Link>
                                            <Link href={`/dashboard/mytransactions`} onClick={closeMenuOnLinkClick} className="btn py-0 px-0 text-decoration-none btn-link mb-2">My Transactions</Link>
                                            <button onClick={handleLogout} className="btn py-0 px-0 text-decoration-none btn-link">Logout</button>
                                        </div>
                                    </div>
                                </div>
                            </>
                        }
                    </div>
                </nav>
            </div>
        </header>
    </>
    );
}

export default DashboardHeader;