"use client";

import { useState, useEffect } from "react";
import { useRouter } from "next/navigation";
import { useDropzone } from "react-dropzone";
import Image from "next/image";
import { Controller, useForm } from "react-hook-form";
import * as yup from 'yup';
import { yupResolver } from '@hookform/resolvers/yup';
import { DatePicker, SelectPicker } from 'rsuite';
import 'rsuite/dist/rsuite.min.css';
import moment from 'moment';
import InputField from "@/components/FormFields/InputField";
import TextAreaField from "../FormFields/TextAreaField";
import { fetchEventCategories, fetchEventSingle } from "@/utils/apiActions";
import { getAuthToken } from "@/utils/auth";

const schema = yup.object({
    event_name: yup.string().required('Event name is required.'),
    event_location: yup.string().required('Event location is required.'),
    category: yup.string().required('Please select category.'),
    startdate: yup.string().required('Please select Event Start Date.'),    
    starttime: yup
    .string()
    .required('Please select Event Start Time.')
    .test(
      'min_start_time',
      'Start time should be greater than current time.',
      function (value) {
        if (!value || value === undefined) {
            return false;
        }
        const { startdate } = this.parent;
        
        if (!moment(startdate).isValid()) {
            return false; // Invalid startdate format
        }

        const [hour, minute] = value.split(':').map(Number);
        const selectedTime = moment(startdate).set({ hour, minute });

        if (moment().isSame(moment(startdate), 'day')) {
            return selectedTime.isAfter(moment());
        }

        if (moment(startdate).isAfter(moment(), 'day')) {
            return true; // Any time is valid
        }
        return false;
      }
    ),
    enddate: yup.string().required('Please select Event End Date.'),
    endtime:yup
    .string()
    .required('Please select Event End Time.')
    .test(
      'min_end_time',
      'End time should be greater than current time.',
      function (value) {
        if (value === undefined) {
          return false;
        }
        const { startdate, enddate, starttime } = this.parent; // Get the `dateEnd` field from the parent object
        const [endHour, endMinute] = value.split(':').map(Number); // Parse `timeEnd`
        const endTime = moment().set({ hour: endHour, minute: endMinute });

        // If dateStart and dateEnd are the same
        if (moment(enddate).format('YYYYMMDD') === moment(startdate).format('YYYYMMDD')) {

            if (starttime) {
                const [startHour, startMinute] = starttime.split(':').map(Number); // Parse `timeStart`
                const startTTime = moment().set({ hour: startHour, minute: startMinute });

                return endTime.isAfter(startTTime); // End time must be greater than start time
            }
            return false; // No start time provided
        }
        else{
            return true;
        }
      }
    ),
}).required();

const EventForm=(props)=>{
    const {isEdit, eventId} = props;
    const [eventData, setEventData] = useState({});
    const [loading, setLoading] = useState(false);
    const [submitLoading, setSubmitLoading] = useState(false);
    const [errorMessage, setErrorMessage] = useState(null);
    const [fileError, setFileError] = useState(null);
    const [catgs, setCatgs] = useState([]);
    const [files ,setFiles] = useState([]);
    const router = useRouter();
    const [isMounted, setIsMounted] = useState(false);

    const { register, handleSubmit, control, setValue, formState: { errors } } = useForm({
        resolver: yupResolver(schema)
    });
    
    const [startDate, setStartDate] = useState(new Date());
    const [endDate, setEndDate] = useState(new Date());

    const token=getAuthToken();

    const disableStartDates = (date) => {
        const today = new Date();
        return date < today.setHours(0, 0, 0, 0);
    };

    const disableEndDate = (date) => {
        if (!startDate){
            const today = new Date();
            return date < today.setHours(0, 0, 0, 0);
        }
        return date < new Date(startDate).setHours(0, 0, 0, 0);
    };
    
    const getCategories=async()=>{
        setLoading(true);
        const [categoriesData] = await Promise.all([
            fetchEventCategories(),
        ]);
        
        let catgOptions = [];
        if(categoriesData.data.length > 0){
            categoriesData.data.forEach(element => {
                catgOptions.push({value:element.id, label:element.category_name});                
            });
        }
        setCatgs(catgOptions);        
        setLoading(false);
    }

    const getCategoriesV2=async()=>{
        setLoading(true);
        const [categoriesData, eventResponse] = await Promise.all([
            fetchEventCategories(),
            fetchEventSingle(eventId)
        ]);
        
        let catgOptions = [];
        if(categoriesData.data.length > 0){
            categoriesData.data.forEach(element => {
                catgOptions.push({value:element.id, label:element.category_name});                
            });
        }
        setCatgs(catgOptions);
        setEventData(eventResponse.data);
        setValue('event_name',eventResponse.data.name);
        setValue('event_location',eventResponse.data.event_location);
        setValue('startdate', new Date(eventResponse.data.start_date));
        setValue('starttime', moment(eventResponse.data.start_date).format('HH:mm:ss'));
        setValue('enddate', new Date(eventResponse.data.end_date));
        setValue('endtime', moment(eventResponse.data.end_date).format('HH:mm:ss'));
        
        if(eventResponse.data.category!==null){
            setValue('category',eventResponse.data.category.id );
        }
        setValue('overview',eventResponse.data.overview);
        setLoading(false);
    }

    const addEventSubmit = async(data)=>{
        setErrorMessage(null);
        const secureHeader = new Headers();
        secureHeader.append("Accept", "application/json");
        secureHeader.append("Accept", "multipart/form-data");
        secureHeader.append("Authorization", `Bearer ${token}`);

        const formdata = new FormData();
        formdata.append("name", data.event_name);
        formdata.append("category_id", data.category);

        let startDate=data.startdate;
        let endDate=data.enddate;
        let starttime=data.starttime;
        let endtime=data.endtime;

        formdata.append("start_date", moment(startDate).format("YYYY-MM-DD")+' '+starttime);
        formdata.append("end_date", moment(endDate).format("YYYY-MM-DD")+' '+endtime);
        formdata.append("event_location", data.event_location);
        formdata.append("overview", data.overview);
        
        if(files.length>0){
            formdata.append("event_photo", files[0].file);
        }

        const requestOptions = {
            method: "POST",
            headers: secureHeader,
            body: formdata,
            redirect: "follow"
        };
        
        setSubmitLoading(true);

        await fetch(`${process.env.NEXT_PUBLIC_API_URL}api/v1/events`, requestOptions)
        .then((response) => response.text())
        .then((result) => {
            let rdata = JSON.parse(result);
            setSubmitLoading(false);
            router.push('/dashboard/events');
        })
        .catch((error) => {
            setSubmitLoading(false);
            setErrorMessage(error);
        });
    }

    const editEventSubmit = async(data)=>{
        setErrorMessage(null);
        const secureHeader = new Headers();
        secureHeader.append("Accept", "application/json");
        secureHeader.append("Accept", "multipart/form-data");
        secureHeader.append("Authorization", `Bearer ${token}`);

        const formdata = new FormData();
        formdata.append("name", data.event_name);
        formdata.append("category_id", data.category);

        let startDate=data.startdate;
        let endDate=data.enddate;
        let starttime=data.starttime;
        let endtime=data.endtime;

        formdata.append("start_date", moment(startDate).format("YYYY-MM-DD")+' '+starttime);
        formdata.append("end_date", moment(endDate).format("YYYY-MM-DD")+' '+endtime);
        formdata.append("event_location", data.event_location);
        formdata.append("overview", data.overview);

        if(files.length>0){
            formdata.append("event_photo", files[0].file);
        }

        const requestOptions = {
            method: "POST",
            headers: secureHeader,
            body: formdata,
            redirect: "follow"
        };
        
        setSubmitLoading(true);

        await fetch(`${process.env.NEXT_PUBLIC_API_URL}api/v1/events/${eventId}`, requestOptions)
        .then((response) => response.text())
        .then((result) => {
            let rdata = JSON.parse(result);
            setSubmitLoading(false);
            router.push('/dashboard/events');
        })
        .catch((error) => {
            setSubmitLoading(false);
            setErrorMessage(error);
        });
    }

    const onSubmit = async (data) => {
        if(isEdit!==undefined && isEdit){
            editEventSubmit(data);
        }
        else{
            addEventSubmit(data);
        }
    };
    

    const {isDragActive, getRootProps, getInputProps, open} = useDropzone({
        noClick: true,
        noKeyboard: true,
        accept: {
            'image/*': ['.png','.jpeg','.jpg','.gif','.webp'],             
        },
        maxFiles:1,
        //maxSize: 2 * 1024 * 1024,
        onDrop: acceptedFiles => {
            let tmpFiles = files;
            acceptedFiles.map((file) =>{
                let tmpObj={};
                tmpObj.file = file;
                tmpObj.preview = URL.createObjectURL(file);
                tmpObj.size = file.size;
                tmpObj.type = file.type;
                tmpObj.name = file.name;                
                tmpFiles.push(tmpObj);                
            });
            setFiles(tmpFiles);
        },
        onDropRejected:(rejections)=>{
            setFileError('File rejected : '+rejections);
        },
    });

    useEffect(()=>{
        if (!isMounted) {
            setIsMounted(true); // Mark as mounted
            return; // Don't run the effect if it's the initial render
        }
        
        if(isEdit!==undefined && isEdit){
            getCategoriesV2();            
        }
        else{
            getCategories();
            setValue('startdate',startDate);
            setValue('enddate',endDate);
        }
    },[isMounted]);

    return(
        <>
            {!loading &&
                <div className="event-form-section">
                    <div className="event-form shadow rounded bg-white mx-auto p-3 p-sm-4">
                        <form onSubmit={handleSubmit(onSubmit)}>
                            <div className="row">
                                <div className="form-group mb-20 col-12">
                                    <InputField
                                        register={register}
                                        disabled={loading}
                                        fieldClass={'form-control shadow-none'}
                                        hideLabel={false}
                                        labelClass={'text-black-200 h6 font-weight-600 mb-2'}
                                        labelText={'Event Name:'}
                                        placeHolder={'Event Name'}
                                        onFieldChange={(e)=>{}}
                                        required={true}
                                        errors={errors}
                                        fieldType={"text"}
                                        fieldId={"event_name"}
                                        fieldName={'event_name'}
                                        autoComplete={""}
                                    />
                                </div>
                                <div className="form-group mb-20 col-12">
                                    <InputField
                                        register={register}
                                        disabled={loading}
                                        fieldClass={'form-control shadow-none'}
                                        hideLabel={false}
                                        labelClass={'text-black-200 h6 font-weight-600 mb-2'}
                                        labelText={'Event Location:'}
                                        placeHolder={'Event Location'}
                                        onFieldChange={(e)=>{}}
                                        required={true}
                                        errors={errors}
                                        fieldType={"text"}
                                        fieldId={"event_location"}
                                        fieldName={'event_location'}
                                        autoComplete={""}
                                    />
                                </div>
                                <div className="form-group mb-20 col-12 col-md-6 col-lg-3">
                                    <label className="text-black-200 h6 font-weight-600 mb-2">Start Date:<sup>*</sup></label>
                                    <Controller
                                        control={control}
                                        name='startdate'
                                        errors={errors}
                                        render={({ field }) => (
                                            <DatePicker
                                                {...field}
                                                className="dtt-field"
                                                value={isEdit ? field.value : startDate}
                                                //defaultValue={isEdit ? field.value : startDate}
                                                onChange={(value) => {
                                                    setStartDate(value);
                                                    setEndDate(value);
                                                    setValue('enddate',value);
                                                    field.onChange(value);                                                        
                                                }}
                                                oneTap
                                                placeholder="Select a date"
                                                format="MM-dd-yyyy"
                                                caretAs={null} 
                                                cleanable={false}
                                                shouldDisableDate={disableStartDates}
                                                block
                                            />
                                        )}
                                    />
                                    {errors && errors['startdate']?.message && (
                                        <p className='font-weight-bold d-block invalid-feedback'>{errors['startdate']?.message}</p>
                                    )}
                                </div>
                                <div className="form-group mb-20 col-12 col-md-6 col-lg-3">
                                    <label className="text-black-200 h6 font-weight-600 mb-2">Start Time:<sup>*</sup></label>
                                    <Controller
                                            control={control}
                                            name='starttime'
                                            errors={errors}
                                            render={({ field }) => (
                                                <DatePicker 
                                                    className="dtt-field"
                                                    placeholder='Start Time *'
                                                    format="hh:mm aa"
                                                    value={field.value ? new Date(`1970-01-01T${field.value}`) : null}
                                                    onChange={(value) => {
                                                        field.onChange(moment(value).format('HH:mm:ss'));
                                                    }}
                                                    caretAs={null} 
                                                    cleanable={false}
                                                    block
                                                />
                                            )}
                                    />
                                    {errors && errors['starttime']?.message && (
                                        <p className='font-weight-bold d-block invalid-feedback'>{errors['starttime']?.message}</p>
                                    )}
                                </div>
                                <div className="form-group mb-20 col-12 col-md-6 col-lg-3">
                                    <label className="text-black-200 h6 font-weight-600 mb-2">End Date:<sup>*</sup></label>
                                    <Controller
                                        control={control}
                                        name='enddate'
                                        errors={errors}
                                        render={({ field }) => (
                                            <DatePicker
                                                {...field}
                                                className="dtt-field"
                                                value={isEdit!==undefined && isEdit ? field.value : endDate}
                                                onChange={(value) => {
                                                    setEndDate(value);
                                                    field.onChange(value);                                                        
                                                }}
                                                oneTap
                                                placeholder="Select a date"
                                                format="MM-dd-yyyy"
                                                caretAs={null} 
                                                cleanable={false}
                                                shouldDisableDate={disableEndDate}
                                                block
                                            />
                                        )}
                                    />
                                    {errors && errors['enddate']?.message && (
                                        <p className='font-weight-bold d-block invalid-feedback'>{errors['enddate']?.message}</p>
                                    )}
                                </div>
                                <div className="form-group mb-20 col-12 col-md-6 col-lg-3">
                                    <label className="text-black-200 h6 font-weight-600 mb-2">End Time:<sup>*</sup></label>
                                    <Controller
                                            control={control}
                                            name='endtime'
                                            errors={errors}
                                            render={({ field }) => (
                                                <DatePicker 
                                                    className="dtt-field"
                                                    placeholder='End Time *'
                                                    format="hh:mm aa" 
                                                    value={field.value ? new Date(`1970-01-01T${field.value}`) : null}
                                                    onChange={(value) => {
                                                        field.onChange(moment(value).format('HH:mm:ss'));
                                                    }}
                                                    caretAs={null} 
                                                    cleanable={false}
                                                    block
                                                />
                                            )}
                                    />
                                    {errors && errors['endtime']?.message && (
                                        <p className='font-weight-bold d-block invalid-feedback'>{errors['endtime']?.message}</p>
                                    )}
                                </div>
                                <div className="form-group mb-20 col-12 col-md-6">
                                    <label className="text-black-200 h6 font-weight-600 mb-2">Category:<sup>*</sup></label>
                                    <Controller
                                        control={control}
                                        name='category'
                                        errors={errors}
                                        render={({ field }) => (
                                            <SelectPicker
                                                className="dtt-field"
                                                data={catgs}
                                                value={field.value}
                                                block
                                                placeholder="Select category"
                                                onChange={(value) => field.onChange(value)} // Handle value change
                                            />
                                        )}
                                    />
                                    {errors && errors['category']?.message && (
                                        <p className='font-weight-bold d-block invalid-feedback'>{errors['category']?.message}</p>
                                    )}
                                </div>
                                <div className="form-group mb-20 col-12 col-md-6">
                                    <label className="text-black-200 h6 font-weight-600 mb-2">Event Picture:</label>
                                    {files.length<=0 ?
                                        <div {...getRootProps({className: `file-dropper`})}>
                                            <input {...getInputProps()} />
                                            <div className={`file-dropper-inner`} onClick={open}>
                                                
                                                <>
                                                    {Object.keys(eventData).length>0 && isEdit!==undefined && isEdit && eventData.event_photo!==null ?
                                                        <Image src={eventData.event_photo.preview_url} alt="Uploaded Image" width={200} height={200} className="mb-2"/>
                                                    :
                                                        <svg className="mb-2" width="72" height="72" viewBox="0 0 72 72" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                            <g opacity="0.8">
                                                                <path d="M57 15V57H15V15H57ZM57 9H15C11.7 9 9 11.7 9 15V57C9 60.3 11.7 63 15 63H57C60.3 63 63 60.3 63 57V15C63 11.7 60.3 9 57 9ZM42.42 35.58L33.42 47.19L27 39.42L18 51H54L42.42 35.58Z" fill="#FF3158" />
                                                            </g>
                                                        </svg>
                                                    }
                                                    {isDragActive ? (
                                                        <p>Drop the image here...</p>
                                                    ) : (
                                                        <p>Drag Images Here Or <span type="button" className="text-primary">Browse</span></p>
                                                    )}
                                                    {fileError!==null && <p className='font-weight-bold d-block invalid-feedback'>{fileError}</p>}
                                                    </>                                                    
                                                
                                            </div>
                                        </div>
                                        :
                                        <div className="file-dropper">
                                            <div className="file-dropper-inner has-preview position-relative">
                                                <Image src={files[0].preview} alt="Uploaded Preview" width={200} height={200} className="mb-2"/>
                                                <button type="button" className="btn btn-primary mx-auto" onClick={()=>{setFiles([])}}>Cancel</button>
                                            </div>
                                        </div>
                                    }
                                </div>
                                <div className="form-group mb-20 col-12">
                                    <TextAreaField
                                        register={register}
                                        disabled={loading}
                                        fieldClass={'form-control shadow-none'}
                                        hideLabel={false}
                                        labelClass={'text-black-200 h6 font-weight-600 mb-2'}
                                        labelText={'Overview:'}
                                        placeHolder={'Overview'}
                                        onFieldChange={(e)=>{}}
                                        required={false}
                                        errors={errors}
                                        fieldType={"text"}
                                        fieldId={"overview"}
                                        fieldName={'overview'}
                                        autoComplete={""}
                                    />
                                </div>
                            </div>
                            <div className="form-group col-12">
                            {!submitLoading ?
                                <button className="btn btn-primary" type="submit">{isEdit!==undefined && isEdit ? 'Update Event' : 'Create Event'}</button>
                                :
                                <div className="btn btn-primary">Submitting...</div>
                            }
                            </div>
                        </form>
                        {errorMessage!==null &&
                            <div className="alert alert-danger mt-3" role="alert">{errorMessage}</div>
                        }
                    </div>
                </div>
            }
        </>
    )
}

export default EventForm;