'use client';
import { useState, useEffect } from "react";
import { useRouter } from "next/navigation";
import { useForm } from "react-hook-form";
import * as yup from 'yup';
import { yupResolver } from '@hookform/resolvers/yup';
import 'rsuite/dist/rsuite.min.css';
import moment from 'moment';
import InputField from "@/components/FormFields/InputField";
import PasswordField from "../FormFields/PasswordField";
import { getAuthToken } from "@/utils/auth";

const schema = yup.object({
    old_password: yup.string().required('Please Enter your password'),
    new_password: yup.string().required('Password is required.').matches(
        /^(?=.*[a-z])(?=.*[A-Z])(?=.*[0-9])(?=.{8,})/,
        "Password must contain at least 8 characters, including one upper case and one number."
    ),
    conf_password: yup.string().oneOf([yup.ref('new_password'), null], 'Passwords must match'),
});

const ChangePasswordForm=()=>{
    const [submitLoading, setSubmitLoading] = useState(false);
    const [isMounted, setIsMounted] = useState(false);
    const [errorMessage, setErrorMessage] = useState(null);
    const [successMessage, setSuccessMessage] = useState(null);
    const router = useRouter();

    const token = getAuthToken();
    const { register, handleSubmit, reset, formState: { errors } } = useForm({
        resolver: yupResolver(schema)
    });

    const onSubmit = async (data) => {
        setErrorMessage(null);
        setSuccessMessage(null);
        setSubmitLoading(true);
        // Prepare form data
        const formData = new FormData();
        const secureHeader = new Headers();
        secureHeader.append("Accept", "application/json");
        secureHeader.append("Accept", "multipart/form-data");
        secureHeader.append("Authorization", `Bearer ${token}`);

        formData.append("current_password", data.old_password);
        formData.append("new_password", data.new_password);
        formData.append("new_password_confirmation", data.conf_password);
        
        // Make the fetch request
        const response = await fetch(`${process.env.NEXT_PUBLIC_API_URL}api/v1/change-password`, {
            method: "POST",
            headers: secureHeader,
            body: formData,
            redirect: "follow"
        });

        setSubmitLoading(false);

        const result = await response.json();
        if (result.status === 1) {
            // Redirect or handle success
            setSuccessMessage(result.message);
            reset();
        } else {
            setErrorMessage(result.message);
        }
    }

    useEffect(()=>{
        if(!isMounted){
            setIsMounted(true);
        }
    },[isMounted]);
    return(
        <>        
            <div className="profile-form-section">
                <div className="container">
                    <div className="col-12 col-md-6 profile-form shadow rounded bg-white p-3 p-sm-4">
                        <form onSubmit={handleSubmit(onSubmit)}>
                            <div className="row">
                                <div className="form-group mb-20 col-12">
                                    <PasswordField
                                        register={register}
                                        disabled={submitLoading}
                                        fieldClass={'form-control shadow-none'}
                                        fieldName={'old_password'}
                                        labelClass={'text-black-200 h6 font-weight-600 mb-2'}
                                        fieldId={'passwordSignIn'}
                                        hideLabel={false}
                                        labelText={'Old Password:'}
                                        placeHolder={'Old Password'}
                                        required={true}
                                        errors={errors}
                                        autoComplete={""}
                                        visibleTrigger={true}
                                    />
                                </div>
                                <div className="form-group mb-20 col-12">
                                    <PasswordField
                                        register={register}
                                        disabled={submitLoading}
                                        fieldClass={'form-control shadow-none'}
                                        fieldName={'new_password'}
                                        labelClass={'text-black-200 h6 font-weight-600 mb-2'}
                                        fieldId={'passwordSignIn'}
                                        hideLabel={false}
                                        labelText={'New Password:'}
                                        placeHolder={'New Password'}
                                        required={true}
                                        errors={errors}
                                        autoComplete={""}
                                        visibleTrigger={true}
                                    />
                                </div>
                                <div className="form-group mb-20 col-12">
                                    <PasswordField
                                        register={register}
                                        disabled={submitLoading}
                                        fieldClass={'form-control shadow-none'}
                                        fieldName={'conf_password'}
                                        labelClass={'text-black-200 h6 font-weight-600 mb-2'}
                                        fieldId={'conf_password'}
                                        hideLabel={false}
                                        labelText={'Confirm Password:'}
                                        placeHolder={'Confirm Password'}
                                        required={true}
                                        errors={errors}
                                        visibleTrigger={true}
                                    />
                                </div>
                            </div>
                            <div className="form-group col-12">
                            {!submitLoading ?
                                <button className="btn btn-primary" type="submit">Submit</button>
                                :
                                <div className="btn btn-primary">Updating...</div>
                            }
                            </div>
                        </form>
                        {errorMessage!==null &&
                            <div className="alert alert-danger mt-3" role="alert">{errorMessage}</div>
                        }
                        {successMessage!==null &&
                            <div className="alert alert-success mt-3" role="alert">{successMessage}</div>
                        }
                    </div>
                </div>
            </div>
        </>
    )
}

export default ChangePasswordForm;