"use client";
import { useState, useEffect } from "react";
import { useRouter } from "next/navigation";
import { useDropzone } from "react-dropzone";
import Image from "next/image";
import { Controller, useForm } from "react-hook-form";
import * as yup from 'yup';
import { yupResolver } from '@hookform/resolvers/yup';
import { DatePicker, SelectPicker } from 'rsuite';
import 'rsuite/dist/rsuite.min.css';
import TextAreaField from "../FormFields/TextAreaField";
import moment from 'moment';
import InputField from "@/components/FormFields/InputField";
import { fetchChallangeCategories, fetchChallangeType, fetchChallangeSingle } from "@/utils/apiActions";
import { getAuthToken } from "@/utils/auth";

const schema = yup.object({
    challange_name: yup.string().required('Challenge name is required.'),
    category: yup.string().required('Please select category.'),
    startdatee: yup.string().required('Please select Challange Start Date.'),
    enddate: yup.string().required('Please select Challange End Date.')
}).required();

const ChallangeForm=(props)=>{
    const {isEdit, challangeId} = props;
    const [challangeData, setChallangeData] = useState({});           
    const [loading, setLoading] = useState(false);
    const [submitLoading, setSubmitLoading] = useState(false);
    const [errorMessage, setErrorMessage] = useState(null);
    const [fileError, setFileError] = useState(null);
    const [catgs, setCatgs] = useState([]);    
    const [types, setTypes] = useState([]);    
    const [files ,setFiles] = useState([]);
    const router = useRouter();
    const [isMounted, setIsMounted] = useState(false);

    const { register, handleSubmit, control, setValue, formState: { errors } } = useForm({
        resolver: yupResolver(schema)
    });
    
    const [startDate, setStartDate] = useState(new Date());
    const [end_date, setend_date] = useState(new Date());

    const token=getAuthToken();

    const disableStartDates = (date) => {
        const today = new Date();
        return date < today.setHours(0, 0, 0, 0);
    };

    const disableend_date = (date) => {
        if (!startDate){
            const today = new Date();
            return date < today.setHours(0, 0, 0, 0);
        }
        return date < new Date(startDate).setHours(0, 0, 0, 0);
    };
    
    const getCategories=async()=>{
        setLoading(true);
        const [categoriesData, challangeData] = await Promise.all([
            fetchChallangeCategories(),
            fetchChallangeType()
        ]);        
        
        let catgOptions = [];
        if(categoriesData.data.length > 0){
            categoriesData.data.forEach(element => {
                catgOptions.push({value:element.id, label:element.name});                
            });
        }
        let typeOptions = [];
        if(challangeData.data.length > 0){
            challangeData.data.forEach(element => {
                typeOptions.push({value:element.id, label:element.name});                
            });
        }
        
        setCatgs(catgOptions);        
        setTypes(typeOptions);        
        setLoading(false);
    }

    const getCategoriesV2=async()=>{
        setLoading(true);
        const [categoriesData,challangeResponse ] = await Promise.all([
            fetchChallangeCategories(),
            fetchChallangeSingle(challangeId)
        ]);
        
        let catgOptions = [];
        if(categoriesData.data.length > 0){
            categoriesData.data.forEach(element => {
                catgOptions.push({value:element.id, label:element.name});                
            });
        }
        setCatgs(catgOptions);           
        setChallangeData(challangeResponse.data);        
        setValue('challange_name',challangeResponse.data.name);
        setValue('startdatee', new Date(challangeResponse.data.start_date));
        setValue('enddate', new Date(challangeResponse.data.end_date));
        setValue('description',challangeResponse.data.description);
        if(challangeResponse.data.category!==null){
            setValue('category',challangeResponse.data.challangecategory_id);
        }        
        if(challangeResponse.data.category!==null){
            setValue('challangeType',challangeResponse.data.challange_type);
            
        }        
        setLoading(false);
    } 
    const addChallangeSubmit = async(data)=>{
        setErrorMessage(null);
        const secureHeader = new Headers();
        secureHeader.append("Accept", "application/json");
        secureHeader.append("Accept", "multipart/form-data");
        secureHeader.append("Authorization", `Bearer ${token}`);

        const formdata = new FormData();
        formdata.append("name", data.challange_name);
        formdata.append("description", data.description);
        formdata.append("challangecategory_id", data.category);        
        formdata.append("challange_type", data.challangeType);        

        let startDate= moment(data.startdatee).format("YYYY-MM-DD");   
        let end_date=moment(data.enddate).format("YYYY-MM-DD");

        formdata.append("startdatee", moment(startDate).format("YYYY-MM-DD HH:mm:ss"));
        formdata.append("enddate", moment(end_date).format("YYYY-MM-DD HH:mm:ss"));
        
        if(files.length>0){
            formdata.append("photo", files[0].file);
        }

        const requestOptions = {
            method: "POST",
            headers: secureHeader,
            body: formdata,
            redirect: "follow"
        };
        
        setSubmitLoading(true);

        await fetch(`${process.env.NEXT_PUBLIC_API_URL}api/v1/challanges`, requestOptions)
        .then((response) => response.text())
        
        .then((result) => {
            let rdata = JSON.parse(result);
            
            setSubmitLoading(false);        
            router.push('/dashboard/wellness-center/challanges');
        })
        .catch((error) => {
            setSubmitLoading(false);
            setErrorMessage(error);
        });
    }

    const editChallangeSubmit = async(data)=>{
        setErrorMessage(null);
        const secureHeader = new Headers();
        secureHeader.append("Accept", "application/json");
        secureHeader.append("Accept", "multipart/form-data");
        secureHeader.append("Authorization", `Bearer ${token}`);

        const formdata = new FormData();
        formdata.append("name", data.challange_name);
        formdata.append("challangecategory_id", data.category);
        formdata.append("challange_type", data.challangeType);
        
        let startDate= moment(data.startdatee).format("YYYY-MM-DD");   
        let end_date=moment(data.enddate).format("YYYY-MM-DD");
        
        formdata.append("startdatee", moment(startDate).format("YYYY-MM-DD HH:mm:ss"));
        formdata.append("enddate", moment(end_date).format("YYYY-MM-DD HH:mm:ss"));
        formdata.append("description", data.description);

        if(files.length>0){
            formdata.append("photo", files[0].file);
        }

        const requestOptions = {
            method: "POST",
            headers: secureHeader,
            body: formdata,
            redirect: "follow"
        };
        
        setSubmitLoading(true);

        await fetch(`${process.env.NEXT_PUBLIC_API_URL}api/v1/challanges/${challangeId}`, requestOptions)
        .then((response) => response.text())
        .then((result) => {
            let rdata = JSON.parse(result);
            setSubmitLoading(false);
            router.push('/dashboard/wellness-center/challanges');
        })
        .catch((error) => {
            setSubmitLoading(false);
            setErrorMessage(error);
        });
    }

    const onSubmit = async (data) => {
        if(isEdit!==undefined && isEdit){
            editChallangeSubmit(data);
        }
        else{
            addChallangeSubmit(data);
        }
    };
    

    const {isDragActive, getRootProps, getInputProps, open} = useDropzone({
        noClick: true,
        noKeyboard: true,
        accept: {
            'image/*': ['.png','.jpeg','.jpg','.gif','.webp'],             
        },
        maxFiles:1,
        //maxSize: 2 * 1024 * 1024,
        onDrop: acceptedFiles => {
            let tmpFiles = files;
            acceptedFiles.map((file) =>{
                let tmpObj={};
                tmpObj.file = file;
                tmpObj.preview = URL.createObjectURL(file);
                tmpObj.size = file.size;
                tmpObj.type = file.type;
                tmpObj.name = file.name;                
                tmpFiles.push(tmpObj);                
            });
            setFiles(tmpFiles);
        },
        onDropRejected:(rejections)=>{
            setFileError('File rejected : '+rejections);
        },
    });

    useEffect(()=>{
        if (!isMounted) {
            setIsMounted(true); // Mark as mounted
            return; // Don't run the effect if it's the initial render
        }
        
        if(isEdit!==undefined && isEdit){
            getCategoriesV2();            
        }
        else{
            getCategories();
            setValue('startdatee',startDate);
            setValue('enddate',end_date);
        }
    },[isMounted]);

    return(
        <>
            {!loading &&
                <div className="event-form-section">
                    <div className="container">
                        <div className="event-form shadow rounded bg-white mx-auto p-3 p-sm-4">
                            <form onSubmit={handleSubmit(onSubmit)}>
                                <div className="row">
                                    <div className="form-group mb-20 col-12">
                                        <InputField
                                            register={register}
                                            disabled={loading}
                                            fieldClass={'form-control shadow-none'}
                                            hideLabel={false}
                                            labelClass={'text-black-200 h6 font-weight-600 mb-2'}
                                            labelText={'Challenge Name:'}
                                            placeHolder={'Challenge Name'}
                                            onFieldChange={(e)=>{}}
                                            required={true}
                                            errors={errors}
                                            fieldType={"text"}
                                            fieldId={"challange_name"}
                                            fieldName={'challange_name'}
                                            autoComplete={""}
                                        />
                                    </div>
                                    <div className="form-group mb-20 col-12 col-md-6 col-lg-3">
                                        <label className="text-black-200 h6 font-weight-600 mb-2">Start Date:<sup>*</sup></label>
                                        <Controller
                                            control={control}
                                            name='startdatee'
                                            errors={errors}
                                            render={({ field }) => (
                                                <DatePicker
                                                    {...field}
                                                    className="dtt-field"
                                                    value={isEdit ? field.value : startDate}
                                                    defaultValue={isEdit ? field.value : startDate}
                                                    onChange={(value) => {
                                                        setStartDate(value);
                                                        setend_date(value);
                                                        setValue('enddate',value);
                                                        field.onChange(value);                                                        
                                                    }}
                                                    oneTap
                                                    placeholder="Select a date"
                                                    format="MM-dd-yyyy"
                                                    caretAs={null} 
                                                    cleanable={false}
                                                    shouldDisableDate={disableStartDates}
                                                    block
                                                />
                                            )}
                                        />
                                        {errors && errors['startdatee']?.message && (
                                            <p className='font-weight-bold d-block invalid-feedback'>{errors['startdatee']?.message}</p>
                                        )}
                                    </div>
                                    <div className="form-group mb-20 col-12 col-md-6 col-lg-3">
                                        <label className="text-black-200 h6 font-weight-600 mb-2">End Date:<sup>*</sup></label>
                                        <Controller
                                            control={control}
                                            name='enddate'
                                            errors={errors}
                                            render={({ field }) => (
                                                <DatePicker
                                                    {...field}
                                                    className="dtt-field"
                                                    value={isEdit!==undefined && isEdit ? field.value : end_date}
                                                    onChange={(value) => {
                                                        setend_date(value);
                                                        field.onChange(value);                                                        
                                                    }}
                                                    oneTap
                                                    placeholder="Select a date"
                                                    format="MM-dd-yyyy"
                                                    caretAs={null} 
                                                    cleanable={false}
                                                    shouldDisableDate={disableend_date}
                                                    block
                                                />
                                            )}
                                        />
                                        {errors && errors['enddate']?.message && (
                                            <p className='font-weight-bold d-block invalid-feedback'>{errors['enddate']?.message}</p>
                                        )}
                                    </div>
                                    <div className="form-group mb-20 col-12 col-md-6">
                                        <label className="text-black-200 h6 font-weight-600 mb-2">Category:<sup>*</sup></label>
                                        <Controller
                                            control={control}
                                            name='category'
                                            errors={errors}
                                            render={({ field }) => (
                                                <SelectPicker
                                                    className="dtt-field"
                                                    data={catgs}
                                                    value={field.value}
                                                    block
                                                    placeholder="Select category"
                                                    onChange={(value) => field.onChange(value)}
                                                />
                                            )}
                                        />
                                        {errors && errors['category']?.message && (
                                            <p className='font-weight-bold d-block invalid-feedback'>{errors['category']?.message}</p>
                                        )}
                                    </div>
                                    <div className="form-group mb-20 col-12 col-md-6">
                                        <label className="text-black-200 h6 font-weight-600 mb-2">Challange Type:</label>
                                        <select className="form-select py-3 px-3 cursor-pointer" {...register("challangeType")} onChange={(e)=>setValue('challangeType', e.target.value, { shouldValidate: true })}>
                                            <option value="onetime" className="cursor-pointer">One Time</option>
                                            <option value="daily" className="cursor-pointer">Daily</option>
                                        </select>
                                    </div>
                                    <div className="form-group mb-20 col-12 col-md-6">
                                        <TextAreaField
                                            register={register}
                                            disabled={loading}
                                            fieldClass={'form-control shadow-none'}
                                            hideLabel={false}
                                            labelClass={'text-black-200 h6 font-weight-600 mb-2'}
                                            labelText={'Description:'}
                                            placeHolder={'Description'}
                                            onFieldChange={(e)=>{}}
                                            required={false}
                                            errors={errors}
                                            fieldType={"text"}
                                            fieldId={"description"}
                                            fieldName={'description'}
                                            autoComplete={""}
                                        />
                                    </div>
                                    <div className="form-group mb-20 col-12 col-md-6">
                                        <label className="text-black-200 h6 font-weight-600 mb-2">Challenge Picture:</label>
                                        {files.length<=0 ?
                                            <div {...getRootProps({className: `file-dropper`})}>
                                                <input {...getInputProps()} />
                                                <div className={`file-dropper-inner`} onClick={open}>
                                                    <>
                                                        {Object.keys(challangeData).length>0 && isEdit!==undefined && isEdit && challangeData.photo!==null ?
                                                            <Image src={challangeData.photo.preview_url} alt="Uploaded Image" width={200} height={200} className="mb-2" style={{
                                                                objectFit: 'cover',
                                                            }}/>
                                                        :
                                                            <svg className="mb-2" width="72" height="72" viewBox="0 0 72 72" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                                <g opacity="0.8">
                                                                    <path d="M57 15V57H15V15H57ZM57 9H15C11.7 9 9 11.7 9 15V57C9 60.3 11.7 63 15 63H57C60.3 63 63 60.3 63 57V15C63 11.7 60.3 9 57 9ZM42.42 35.58L33.42 47.19L27 39.42L18 51H54L42.42 35.58Z" fill="#FF3158" />
                                                                </g>
                                                            </svg>
                                                        }
                                                        {isDragActive ? (
                                                            <p>Drop the image here...</p>
                                                        ) : (
                                                            <p>Drag Images Here Or <span type="button" className="text-primary">Browse</span></p>
                                                        )}
                                                        {fileError!==null && <p className='font-weight-bold d-block invalid-feedback'>{fileError}</p>}
                                                    </>                                                    
                                                </div>
                                            </div>
                                            :
                                            <div className="file-dropper">
                                                <div className="file-dropper-inner has-preview position-relative">
                                                    <Image src={files[0].preview} alt="Uploaded Preview" width={200} height={200} className="mb-2"/>
                                                    <button type="button" className="btn btn-primary mx-auto" onClick={()=>{setFiles([])}}>Cancel</button>
                                                </div>
                                            </div>
                                        }
                                    </div>
                                </div>
                                <div className="form-group col-12">
                                {!submitLoading ?
                                    <button className="btn btn-primary" type="submit">{isEdit!==undefined && isEdit ? 'Update Challanges' : 'Create Challenges'}</button>
                                    :
                                    <div className="btn btn-primary">Updating...</div>
                                }
                                </div>
                            </form>
                            {errorMessage!==null &&
                                <div className="alert alert-danger mt-3" role="alert">{errorMessage}</div>
                            }
                        </div>
                    </div>
                </div>
            }
        </>
    )
}

export default ChallangeForm;