'use client';

import {useEffect, useState, useRef } from 'react';
import Image from 'next/image';
import Link from 'next/link';
import moment from 'moment';
import { confirmAlert } from 'react-confirm-alert';
import 'react-confirm-alert/src/react-confirm-alert.css';
import ShareEventToConnectionPopup from '../Popups/ShareEventToConnectionPopup';

const EventCard=(props)=>{
    const {eventItem, dispatch, deleteEventAction, getEvents, isShared, token} = props;
    const dropdownRef = useRef(null);
    const [showSharePopup, setShowSharePopup] = useState(false);
    const [isDropdownOpen, setDropdownOpen] = useState(false);
    const toggleDropdown = (event) => {
        event.preventDefault();
        setDropdownOpen(!isDropdownOpen);
    };
    const handleClickOutside = (event) => {
        if (dropdownRef.current && !dropdownRef.current.contains(event.target)) {
          setDropdownOpen(false);
        }
    };
    useEffect(() => {
        if (isDropdownOpen) {
            document.addEventListener("mousedown", handleClickOutside);
        } else {
            document.removeEventListener("mousedown", handleClickOutside);
        }
    
        return () => {
            document.removeEventListener("mousedown", handleClickOutside);
        };
    }, [isDropdownOpen]);
    const deleteEventCallback=async()=>{
        const secureHeader = new Headers();
        secureHeader.append("Authorization", `Bearer ${token}`);

        const requestOptions = {
            method: "DELETE",
            headers: secureHeader,
            redirect: "follow"
        };

        await fetch(`${process.env.NEXT_PUBLIC_API_URL}api/v1/events/${eventItem.id}`, requestOptions)
        .then((response) => response.text())
        .then((result) => {
            dispatch(deleteEventAction({id:eventItem.id}));
            getEvents();
        })
        .catch((error) => console.error(error));
    }
    const shareEvent=()=>{
        setShowSharePopup(true);
    }

    const deleteEvent = () =>{
        confirmAlert({
            customUI: ({ onClose }) => {
                return (
                <div className='delete-confirm-ui p-3 w-100 mx-auto'>
                    <div className="fs-5 fw-bold border-bottom border-grey pb-2 mb-2 text-center">Are you sure?</div>
                    <p className="fs-6 mb-4 pb-4 text-center">Are you sure you want to delete this event?</p>
                    <div className="d-flex justify-content-end gap-3 border-top brder-gray pt-3">
                        <button className="btn btn-primary btn-sm"
                        onClick={() => { 
                            deleteEventCallback();
                            onClose();
                        }}
                        >
                        Yes
                        </button>
                        <button className="btn btn-outline-primary btn-sm" onClick={onClose}>No</button>                        
                    </div>
                </div>
                );
            }
        });
    }

    return(
        <>
        <div className="event-card w-100 h-100 p-2 p-md-3 position-relative d-flex flex-column">
            <div className="d-flex justify-content-between mb-3">
                <h4 className="event-card-title text-primary flex-grow-1 pe-3"><Link href={`/dashboard/events/${eventItem.id}`} className='text-primary'>{eventItem.name}</Link></h4>
                {!isShared &&
                    <div className="event-actions challange-actions d-flex gap-1 justify-content-end" ref={dropdownRef}>
                    <button type='button' onClick={toggleDropdown} className="action-cta justify-content-end bottom-0"><i className="ti-more-alt"></i></button>
                    {isDropdownOpen && (
                        <div className="position-absolute right-0 z-2 mt-4 w-48 bg-white border border-gray-200 rounded shadow-lg">
                            <ul className='list-actions p-3 mb-0'>
                                <li className="pb-3 hover:bg-gray-100 cursor-pointer"><Link href={`/dashboard/events/edit/${eventItem.id}`} className='text-dark text-decoration-none'><i className="ti-pencil me-2"></i> Edit Event</Link></li>
                                <li className="pb-3 hover:bg-gray-100 cursor-pointer"><button onClick={shareEvent} className='text-dark text-decoration-none'><i className="ti-calendar me-2"></i> Share Event</button></li>
                                <li className="hover:bg-gray-100 cursor-pointer"><button onClick={deleteEvent} className='text-dark text-decoration-none'><i className="ti-close me-2"></i> Delete Event</button></li>
                            </ul>
                            
                        </div>
                    )}
                    </div>
                }
            </div>
            <div className="d-flex flex-grow-1">
                {eventItem?.event_photo!==null &&
                <div className="image-wrapper position-relative">
                    <Image
                        src={eventItem?.event_photo?.preview_url}
                        alt={eventItem.name}
                        //sizes="100vw"
                        fill
                        style={{
                            objectFit: 'cover', // cover, contain, none
                        }}
                        priority
                    />
                </div>
                }
                <div className={`flex-grow-1 d-flex flex-column justify-content-between event-meta ${eventItem?.event_photo!==null && 'ps-1 ps-lg-3'}`}>
                    <div>
                        <div className='d-flex  mb-1'><i className='ti-calendar pt-1'></i>
                            <span className='px-2'>
                                {moment(eventItem.start_date).format('MDDY')!==moment(eventItem.end_date).format('MDDY') ?
                                    <>
                                    {moment(eventItem.start_date).format('MMM DD,Y')} - {moment(eventItem.end_date).format('MMM DD,Y')}
                                    </>
                                    :
                                    <>
                                        {moment(eventItem.start_date).format('MMM DD,Y')}
                                    </>
                                }
                            </span>
                        </div>
                        <div className='d-flex align-items-center mb-1'><i className='ti-time'></i>
                            <span className='px-2'>
                                {moment(eventItem.start_date).format('HH:mm A')} - {moment(eventItem.end_date).format('HH:mm A')}
                            </span>
                        </div>
                        <div className='d-flex align-items-center mb-1'><i className='ti-location-pin'></i><span className='px-2'>{eventItem.event_location}</span></div>                        
                    </div>
                    <div className='d-flex justify-content-lg-start mt-2'><Link href={`/dashboard/events/${eventItem.id}`} className='btn btn-primary btn-tiny text-uppercase fw-bold'>Read more</Link></div>
                </div>
            </div>
        </div>
        {showSharePopup &&
            <ShareEventToConnectionPopup setShowPopup={setShowSharePopup} eventItem={eventItem} token={token} />
        }
        </>
    )
}

export default EventCard