'use client';
import { useState, useEffect, useRef } from 'react';
import Image from 'next/image';
import Link from 'next/link';
import moment from 'moment';
import { confirmAlert } from 'react-confirm-alert';
import 'react-confirm-alert/src/react-confirm-alert.css';
import ImageNotFound from '../../../public/images/logo.png';
import { useRouter } from 'next/navigation';

const ChallangeCard = ({ challangeItem, token, isShared }) => {
    const [isDropdownOpen, setDropdownOpen] = useState(false);
    const [challanges, setChallanges] = useState([]); // Local state for challenges
    const dropdownRef = useRef(null);
    const router = useRouter();

    useEffect(() => {
        if (isDropdownOpen) {
            document.addEventListener("mousedown", handleClickOutside);
        } else {
            document.removeEventListener("mousedown", handleClickOutside);
        }

        return () => {
            document.removeEventListener("mousedown", handleClickOutside);
        };
    }, [isDropdownOpen]);

    if (challangeItem.status === "completed") {
        return null;
    }

    const toggleDropdown = (event) => {
        event.preventDefault();
        setDropdownOpen(!isDropdownOpen);
    };

    const handleClickOutside = (event) => {
        if (dropdownRef.current && !dropdownRef.current.contains(event.target)) {
            setDropdownOpen(false);
        }
    };

    const updateChallanges = (updatedChallangeId) => {
        setChallanges((prevChallanges) => prevChallanges.filter(ch => ch.id !== updatedChallangeId));
    };

    const deleteChallangeCallback = async () => {
        const secureHeader = new Headers();
        secureHeader.append("Authorization", `Bearer ${token}`);

        const requestOptions = {
            method: "DELETE",
            headers: secureHeader,
            redirect: "follow"
        };

        await fetch(`${process.env.NEXT_PUBLIC_API_URL}api/v1/challanges/${challangeItem.id}`, requestOptions)
            .then(() => {
                updateChallanges(challangeItem.id);
                window.location.reload();
            })
            .catch((error) => console.error(error));
    };

    const completeChallangeCallback = async () => {
        const secureHeader = new Headers();
        secureHeader.append("Authorization", `Bearer ${token}`);
        const formdata = new FormData();
        formdata.append("challange_id", challangeItem.id);
        formdata.append("status", 'complete');

        const requestOptions = {
            method: "POST",
            headers: secureHeader,
            body: formdata,
            redirect: "follow"
        };

        await fetch(`${process.env.NEXT_PUBLIC_API_URL}api/v1/userchallanges/handle`, requestOptions)
            .then(() => {
                updateChallanges(challangeItem.id);
                window.location.reload();
            })
            .catch((error) => console.error(error));
    };

    const deleteChallange = () => {
        confirmAlert({
            customUI: ({ onClose }) => {
                return (
                    <div className='delete-confirm-ui p-3 w-100 mx-auto'>
                        <div className="fs-5 fw-bold border-bottom border-grey pb-2 mb-2 text-center">Are you sure?</div>
                        <p className="fs-6 mb-4 pb-4 text-center">Are you sure you want to delete this challange?</p>
                        <div className="d-flex justify-content-end gap-3 border-top brder-gray pt-3">
                            <button className="btn btn-primary btn-sm"
                                onClick={() => {
                                    deleteChallangeCallback();
                                    onClose();
                                }}
                            >
                                Yes
                            </button>
                            <button className="btn btn-outline-primary btn-sm" onClick={onClose}>No</button>
                        </div>
                    </div>
                );
            }
        });
    };

    const cancelParticipateCallback = async () => {
        const secureHeader = new Headers();
        secureHeader.append("Authorization", `Bearer ${token}`);
        const formdata = new FormData();
        formdata.append("challange_id", challangeItem.id);
        formdata.append("status", 'cancel');

        const requestOptions = {
            method: "POST",
            headers: secureHeader,
            body: formdata,
            redirect: "follow"
        };

        await fetch(`${process.env.NEXT_PUBLIC_API_URL}api/v1/userchallanges/handle`, requestOptions)
            .then(() => {
                window.location.reload();
                updateChallanges(challangeItem.id);
            })
            .catch((error) => console.error(error));
    };

    const cancelParticipate = () => {
        confirmAlert({
            customUI: ({ onClose }) => {
                return (
                    <div className='delete-confirm-ui p-3 w-100 mx-auto'>
                        <div className="fs-5 fw-bold border-bottom border-grey pb-2 mb-2 text-center">Are you sure?</div>
                        <p className="fs-6 mb-4 pb-4 text-center">Are you sure you want to cancel this participation?</p>
                        <div className="d-flex justify-content-end gap-3 border-top brder-gray pt-3">
                            <button className="btn btn-primary btn-sm"
                                onClick={() => {
                                    cancelParticipateCallback();
                                    onClose();
                                }}>
                                Yes
                            </button>
                            <button className="btn btn-outline-primary btn-sm" onClick={onClose}>No</button>
                        </div>
                    </div>
                );
            }
        });
    };

    const completeChallange = () => {
        confirmAlert({
            customUI: ({ onClose }) => {
                return (
                    <div className='delete-confirm-ui p-3 w-100 mx-auto'>
                        <div className="fs-5 fw-bold border-bottom border-grey pb-2 mb-2 text-center">Are you sure?</div>
                        <p className="fs-6 mb-4 pb-4 text-center">Are you sure you want to complete this challange?</p>
                        <div className="d-flex justify-content-end gap-3 border-top brder-gray pt-3">
                            <button className="btn btn-primary btn-sm"
                                onClick={() => {
                                    completeChallangeCallback();
                                    onClose();
                                }}>
                                Yes
                            </button>
                            <button className="btn btn-outline-primary btn-sm" onClick={onClose}>No</button>
                        </div>
                    </div>
                );
            }
        });
    };

    const viewChallange = () => {
        confirmAlert({
            customUI: ({ onClose }) => {
                return (
                    <div className='view-challanges-ui p-4 w-100 mx-auto'>
                        <div className='d-flex gap-3'>
                            <div className='position-relative'>
                                {challangeItem?.photo?.preview_url ? (
                                    <Image
                                        src={challangeItem?.photo?.preview_url}
                                        alt={challangeItem.name}
                                        width={100}
                                        height={100}
                                        className='rounded-2'
                                        style={{
                                            objectFit: 'cover',
                                        }}
                                    />
                                ) : (
                                    <div className='d-flex flex-column align-items-center w-100 h-100 justify-content-center bg-light'>
                                        <Image
                                            src={ImageNotFound}
                                            width={120}
                                            height={120}
                                            alt=''
                                            style={{
                                                objectFit: 'cover',
                                            }}
                                        />
                                    </div>
                                )}
                            </div>
                            <div className='w-100'>
                                <div>
                                    <h5 className="fw-bold border-grey mb-2">{challangeItem.name}</h5>
                                    <div>
                                        <i className="ti-calendar me-2"></i>
                                        {moment(challangeItem.start_date).format('MDDY') !== moment(challangeItem.end_date).format('MDDY') ? (
                                            <>
                                                {moment(challangeItem.start_date).format('MMM DD,Y')} TO {moment(challangeItem.end_date).format('MMM DD,Y')}
                                            </>
                                        ) : (
                                            <>
                                                {moment(challangeItem.start_date).format('MMM DD,Y')}
                                            </>
                                        )}
                                    </div>
                                </div>

                                <div className='mb-0'>
                                    <span>
                                        <p><b>Category</b>: {challangeItem.challangecategory_name}</p>
                                    </span>
                                    <span>
                                        <p><b>Type</b>: {challangeItem.challange_type}</p>
                                    </span>
                                </div>
                                {challangeItem.challange_type === 'daily' &&
                                    <div className='bg-primary d-flex align-items-center px-2 py-1 rounded-3 text-white mt-1'>
                                        <span>
                                            <p><b>Total Days</b>: <b className='bg-white text-primary px-2 rounded-pill'>{challangeItem.total_days}</b></p>
                                        </span>
                                        <span className='ms-auto'>
                                            <p><b>Completed Days</b>: <b className='bg-white text-primary px-2 rounded-pill'>{challangeItem.completed_days}</b></p>
                                        </span>
                                    </div>
                                }
                            </div>
                        </div>
                        <div className='border-top mt-2 pt-2'>
                            <h5 className="fw-bold border-grey">Overview</h5>
                            <p>{challangeItem.description}</p>
                        </div>
                        <div className='position-absolute right-0 view-challanges-inner'>
                            <i className="ti-close" onClick={onClose}></i>
                        </div>
                    </div>
                );
            }
        });
    };

    return (
        <div className='challange-card w-100 h-100 position-relative d-flex flex-column'>
            <div className="card h-100">
                <div className="card-body">
                    <div className='position-relative'>
                        <div className="image-wrapper position-relative">
                            {challangeItem?.photo?.original_url ? (
                                <Image
                                    src={challangeItem.photo.original_url}
                                    alt={challangeItem.name}
                                    fill
                                    style={{
                                        objectFit: 'cover',
                                        objectPosition: 'top',
                                    }}
                                />
                            ) : (
                                <div className='d-flex flex-column align-items-center w-100 h-100 justify-content-center bg-light'>
                                    <Image
                                        src={ImageNotFound}
                                        width={120}
                                        height={120}
                                        alt=''
                                        style={{
                                            objectFit: 'cover',
                                        }}
                                    />
                                </div>
                            )}
                            {challangeItem.status === 'complete' ?
                                <div className='success-msg-completed w-100' style={{
                                    zIndex: '2'
                                }}>
                                    <span>{challangeItem.status}</span>
                                </div>
                                : challangeItem.status === 'failed' ?
                                
                                <div className='failed-msg w-100' style={{
                                    zIndex: '2'
                                }}>
                                    <span>{challangeItem.status}</span>
                                </div>
                                :
                                <div className='inprogress-msg-completed w-100' style={{
                                    zIndex: '2'
                                }}>
                                    <span>{challangeItem.status}</span>
                                </div>
                            }
                        </div>
                        <div className='position-absolute bottom-0 w-100 d-flex justify-content-between challanges-categoryType'>
                            <span className='fw-bold'>
                                {challangeItem.challangecategory_name}
                            </span>
                        </div>
                    </div>
                    <h4 className="event-card-title text-primary flex-grow-1 mt-3 text-start">
                        <button onClick={viewChallange} className='text-primary fw-bold'>{challangeItem.name}</button>
                        <span onClick={viewChallange} className='card_Link'></span>
                    </h4>
                    <p className="mb-3 text-truncate">
                        {challangeItem.description}
                    </p>
                    <div>
                        <span>
                            <p className='text-dark mb-2'><b>Type</b>: {challangeItem.challange_type}</p>
                        </span>
                        <div className='mb-2'>
                            <i className="ti-calendar me-2"></i>
                            {moment(challangeItem.start_date).format('MDDY') !== moment(challangeItem.end_date).format('MDDY') ? (
                                <>
                                    {moment(challangeItem.start_date).format('MMM DD,Y')} TO {moment(challangeItem.end_date).format('MMM DD,Y')}
                                </>
                            ) : (
                                <>
                                    {moment(challangeItem.start_date).format('MMM DD,Y')}
                                </>
                            )}
                        </div>
                    </div>
                    {!isShared && (
                        <>
                        {(challangeItem.status !== 'complete' && challangeItem.status !== 'failed') &&
                            <div className="event-actions challange-actions d-flex gap-1 justify-content-end" ref={dropdownRef}>
                                <button type='button' onClick={toggleDropdown} className="action-cta justify-content-end position-absolute bottom-0 mb-20"><i className="ti-more-alt"></i></button>
                                {isDropdownOpen && (
                                    <div className="position-absolute right-0 z-2 mt-2 w-48 bg-white border border-gray-200 rounded shadow-lg">
                                        <ul className='list-actions p-3 mb-0'>
                                            {challangeItem.is_admin_challenge !== true && (
                                                <>
                                                    <li className="pb-3 hover:bg-gray-100 cursor-pointer">
                                                        <Link href={`/dashboard/wellness-center/challanges/edit/${challangeItem.id}`} className='text-dark text-decoration-none'>
                                                            <i className="ti-pencil me-2"></i> Edit Challange
                                                        </Link>
                                                    </li>
                                                    <li className=" hover:bg-gray-100 cursor-pointer">
                                                        <button onClick={deleteChallange} className='text-dark text-decoration-none'>
                                                            <i className="ti-trash me-2"></i> Delete Challange
                                                        </button>
                                                    </li>
                                                </>
                                            )}
                                            {challangeItem.is_admin_challenge !== false && (
                                                <>
                                                    <li className="hover:bg-gray-100 cursor-pointer">
                                                        <button onClick={cancelParticipate} className='text-dark text-decoration-none'>
                                                            <i className="ti-close me-2"></i> Cancel Participation
                                                        </button>
                                                    </li>
                                                </>
                                            )}
                                            {challangeItem.challange_type === 'onetime' ?
                                                <li className="hover:bg-gray-100 cursor-pointer pt-3">
                                                    <button onClick={completeChallange} className="text-dark text-decoration-none">
                                                    <i className="ti-check me-2"></i> 
                                                    {challangeItem.challange_type === "onetime" && "Completed Challenges"}
                                                    </button>
                                                </li>
                                                :
                                                <li className="hover:bg-gray-100 cursor-pointer">
                                                    {new Date() >= new Date(challangeItem.start_date) && new Date() <= new Date(challangeItem.end_date) && (
                                                        <button onClick={completeChallange} className="pt-3 text-dark text-decoration-none">
                                                        <i className="ti-check me-2"></i> 
                                                        {challangeItem.challange_type === "daily" && "Completed Today"}
                                                        </button>
                                                    )}
                                                </li>
                                            }

                                        </ul>
                                    </div>
                                )}
                            </div>
                        }
                        </>
                    )}
                </div>
            </div>
        </div>
    );
};

export default ChallangeCard;
