"use client"
import { useState, useEffect } from "react";
import { useDispatch, useSelector } from "react-redux";
import Image from "next/image";
import Link from "next/link";
import { fetchKeepSakes } from "@/utils/apiActions";
import { deleteSkeepsakesAction } from "@/store/slices/keepstakesSlice";
import { confirmAlert } from "react-confirm-alert";
import "react-confirm-alert/src/react-confirm-alert.css";
import { getAuthToken } from "@/utils/auth";
import Audio from '../../../../../public/audio-ic.svg';
import Video from '../../../../../public/video-ic.svg';
import Pdf from '../../../../../public/pdf-ic.svg';
import Document from '../../../../../public/doc-ic.svg';

const KeepSakesPage = () => {
    const [loading, setLoading] = useState(true);
    const [keepsake, setKeepsake] = useState([]);
    const [selectedImage, setSelectedImage] = useState(null);
    const [imageIndex, setImageIndex] = useState(0);
    const token = getAuthToken();
    const dispatch = useDispatch();

    const userObj = useSelector(state => state.user);
    const currentUser = userObj.user;

    const getResources = async () => {
        try {
            const resourcesData = await fetchKeepSakes();
            setKeepsake(resourcesData.data);
        } catch (error) {
            console.error("Error fetching resources:", error);
        } finally {
            setLoading(false);
        }
    };

    useEffect(() => {
        getResources();
    }, []);

    const deleteKeepsakesCallback = async (keepsakeId) => {
        const secureHeader = new Headers();
        secureHeader.append("Authorization", `Bearer ${token}`);

        const requestOptions = {
            method: "DELETE",
            headers: secureHeader,
            redirect: "follow",
        };

        await fetch(`${process.env.NEXT_PUBLIC_API_URL}api/v1/keeps-sakes/${keepsakeId}`, requestOptions)
            .then((response) => response.text())
            .then((result) => {
                dispatch(deleteSkeepsakesAction({ id: keepsakeId }));
                setKeepsake((prev) => prev.filter((item) => item.id !== keepsakeId));
            })
            .catch((error) => console.error(error));
    };

    const deleteKeepsakes = (keepsakeId) => {
        confirmAlert({
            customUI: ({ onClose }) => {
                return (
                    <div className="delete-confirm-ui p-3 w-100 mx-auto">
                        <div className="fs-5 fw-bold border-bottom border-grey pb-2 mb-2 text-center">
                            Are you sure?
                        </div>
                        <p className="fs-6 mb-4 pb-4 text-center">
                            Are you sure you want to delete this keepsake?
                        </p>
                        <div className="d-flex justify-content-end gap-3 border-top border-gray pt-3">
                            <button
                                className="btn btn-primary btn-sm"
                                onClick={() => {
                                    deleteKeepsakesCallback(keepsakeId);
                                    onClose();
                                }}
                            >
                                Yes
                            </button>
                            <button className="btn btn-outline-primary btn-sm" onClick={onClose}>
                                No
                            </button>
                        </div>
                    </div>
                );
            },
        });
    };

    const openImageModal = (url, index) => {
        setSelectedImage(url);
        setImageIndex(index);
    };

    const closeImageModal = () => {
        setSelectedImage(null);
    };

    const handlePreviousImage = () => {
        setImageIndex((prevIndex) => (prevIndex === 0 ? keepsake.length - 1 : prevIndex - 1));
        setSelectedImage(keepsake[imageIndex === 0 ? keepsake.length - 1 : imageIndex - 1]?.photo.original_url);
    };

    const handleNextImage = () => {
        setImageIndex((prevIndex) => (prevIndex === keepsake.length - 1 ? 0 : prevIndex + 1));
        setSelectedImage(keepsake[imageIndex === keepsake.length - 1 ? 0 : imageIndex + 1]?.photo.original_url);
    };

    useEffect(() => {
        if (selectedImage) {
            document.body.style.overflow = "hidden";
        } else {
            document.body.style.overflow = "auto";
        }
        if(Object.keys(currentUser).length > 0){
            if(!currentUser.subscription_status){
                window.location.href = "/dashboard/subscriptionplan";
            }
        }
    }, [selectedImage, currentUser]);



    const getFileType = (url) => {
        const extension = url?.split(".").pop().toLowerCase();
        if (["mp3", "wav", "ogg"].includes(extension)) return "audio";
        if (["mp4", "mov", "avi"].includes(extension)) return "video";
        if (["pdf"].includes(extension)) return "pdf";
        if (["doc", "docx"].includes(extension)) return "word";
        return "image";
    };
    
    const getIconForFileType = (type) => {
        switch (type) {
            case "audio":
                return <Image src={Audio} alt="" className="w-25 h-25"></Image>;
            case "video":
                return <Image src={Video} alt="" className="w-25 h-25"></Image>;
            case "pdf":
                return <Image src={Pdf} alt="" className="w-25 h-25"></Image>;
            case "word":
                return <Image src={Document} alt="" className="w-25 h-25"></Image>;
            default:
                return null;
        }
    };
    return (
        <>
            <div className="d-flex justify-content-between align-items-center gap-2 mb-4 border-bottom border-gray pb-2">
                <div className="col-auto">
                    <h2 className="section-title mb-0">Keepsakes</h2>
                </div>
                <div className="col-auto data-action-bar">
                    <Link
                        href={"/dashboard/friends-family/keepsakes/create"}
                        type="button"
                        className="data-action-cta data-action-cta-text shadow-sm"
                    >
                        <i className="ti-plus me-1"></i> Add New
                    </Link>
                </div>
            </div>
            {loading ? (
                <div className="flex justify-center items-center h-screen text-center">
                    <div className="lds-ripple">
                        <div></div>
                        <div></div>
                    </div>
                </div>
            ) : keepsake.length === 0 ? (
                <div className="text-center text-muted">
                    <h5 className="fw-normal">No Records Found</h5>
                </div>
            ) : (
                <>
                    <div className="row justify-content-center">
                    {keepsake.map((keepsake, index) => {
                        const fileType = getFileType(keepsake?.photo.original_url);
                        return (
                            <div className="col-12 col-sm-6 col-md-4 col-lg-3 mb-4" key={index}>
                                <div className="keepsafe-card position-relative ratio ratio-4x3 overflow-hidden">
                                    <a
                                        href="#"
                                        onClick={() => openImageModal(keepsake?.photo.original_url, index)}
                                        className="image-popup d-flex align-items-center justify-content-center"
                                    >
                                        {fileType === "image" ? (
                                            <Image
                                                src={keepsake?.photo.original_url}
                                                alt="Keepsake"
                                                fill
                                                className="rounded object-fit-cover"
                                            />
                                        ) : (
                                            <div className="d-flex align-items-center justify-content-center flex-col w-full h-full bg-gray-200 rounded">
                                                {getIconForFileType(fileType)}
                                            </div>
                                        )}
                                    </a>
                                    <div className="keepsafe-wrapper position-absolute bottom-0 left-0 w-100 p-3">
                                        <div className="d-flex gap-2 align-items-center justify-content-between">
                                            <div className="text-white">{keepsake.title}</div>
                                            <div className="d-flex gap-2 keepsakes-ic">
                                                <button type="button" onClick={() => deleteKeepsakes(keepsake.id)}>
                                                    <i className="ti-trash"></i>
                                                </button>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        );
                    })}
                    </div>
                    {selectedImage && (
                        <div className="image-modal position-fixed top-0 left-0 w-100 h-100 bg-black bg-opacity-75 d-flex justify-content-center align-items-center">
                            <div className="image-modal-content position-relative">
                                {getFileType(selectedImage) === "image" ? (
                                    <Image
                                        src={selectedImage}
                                        height={500}
                                        width={500}
                                        alt="Full-screen view"
                                        className="rounded w-100 object-fit-cover"
                                        onClick={(e) => e.stopPropagation()}
                                    />
                                ) : getFileType(selectedImage) === "audio" ? (
                                    <div className="audio-preview w-100">
                                        <audio controls autoPlay className="w-100">
                                            <source src={selectedImage} type="audio/mpeg" />
                                            Your browser does not support the audio element.
                                        </audio>
                                    </div>
                                ) : getFileType(selectedImage) === "video" ? (
                                    <div className="video-preview w-100">
                                        <video controls autoPlay className="w-100">
                                            <source src={selectedImage} type="video/mp4" />
                                            Your browser does not support the video element.
                                        </video>
                                    </div>
                                ) : getFileType(selectedImage) === "pdf" ? (
                                    <div className="pdf-preview w-100">
                                        <iframe
                                            src={selectedImage}
                                            title="PDF Preview"
                                            className="w-100"
                                            height="500"
                                        />
                                    </div>
                                ) : getFileType(selectedImage) === "word" ? (
                                    <div className="word-preview w-100">
                                        <iframe
                                            src={`https://docs.google.com/gview?url=${selectedImage}&embedded=true`}
                                            title="Word Preview"
                                            className="w-100"
                                            height="500"
                                        />
                                    </div>
                                ) : null}

                                <div className="modal-controls position-absolute top-50 start-0 translate-middle-y d-flex align-items-center gap-3">
                                    <button className="btn btn-light left-btn" onClick={handlePreviousImage}>
                                        <i className="ti-arrow-left"></i>
                                    </button>
                                    <button className="btn btn-light right-btn" onClick={handleNextImage}>
                                        <i className="ti-arrow-right"></i>
                                    </button>
                                </div>

                                <button className="close-btn btn btn-light position-absolute" onClick={closeImageModal}>
                                    <i className="ti-close"></i>
                                </button>
                            </div>
                        </div>
                    )}

                </>
            )}
        </>

    );
};

export default KeepSakesPage;
