'use client';
import { useState, useEffect } from "react";
import { useDispatch, useSelector } from "react-redux";
import { usePathname } from "next/navigation";
import ReactPaginate from "react-paginate";
import { fetchEventAll } from "@/utils/apiActions";
import EventCard from "@/components/Cards/EventCard";
import { fetchEventsAction, deleteEventAction } from "@/store/slices/eventsSlice";
import { getAuthToken } from "@/utils/auth";

const MyEventsPage = () => {
    const [loading, setLoading] = useState(false);
    const eventsListObj = useSelector(state => state.events);
    const eventsList = eventsListObj.eventsData;
    const [pageOffset, setPageOffset] = useState(0);
    const [pageCount, setPageCount] = useState(0);
    const [selectedCategory, setSelectedCategory] = useState('all');
    const perPage = 10;
    const dispatch = useDispatch();
    const token = getAuthToken();
    const [isMounted, setIsMounted] = useState(false);
    const pathname = usePathname();

    const userObj = useSelector(state => state.user);
    const currentUser = userObj.user;

    const getEvents = async () => {
        setLoading(true);
        const [eventsData] = await Promise.all([
            fetchEventAll(selectedCategory, pageOffset + 1, perPage),
        ]);

        if (eventsData.status === 1) {
            dispatch(fetchEventsAction(eventsData.data));
            setPageCount(eventsData.pagination.last_page);
        }
        setLoading(false);
    };

    const handlePageChange = (event) => {
        setPageOffset(event.selected);
    };

    const internalMenu = [
        { name: 'My Events', href: '/dashboard/events' },
        { name: 'Shared Events', href: '/dashboard/events/shared' },
        { name: 'Create Event', href: '/dashboard/events/create' }
    ];

    const isActive = (href) => {
        if (href === '/dashboard/events') {
            return pathname === href;
        }

        return pathname === href || pathname.startsWith(href);
    };

    useEffect(() => {
        if (!isMounted) {
            setIsMounted(true);
            return;
        }
        getEvents();

        if (Object.keys(currentUser).length > 0) {
            if (!currentUser.subscription_status) {
                window.location.href = "/dashboard/subscriptionplan";
            }
        }
    }, [isMounted, pageOffset, perPage, currentUser]);

    return (
        <>
            {loading ? (
                <div style={{ width: '100%', height: '300px', position: 'relative' }}>
                    <div className="css-loader position-absolute w-100 h-100 top-0 left-0">
                        <div className="lds-ripple"><div></div><div></div></div>
                    </div>
                </div>
            ) : (
                <>
                    {eventsList.length > 0 ? (
                        <>
                            <div className="d-flex justify-content-between align-items-start align-items-md-center mb-4 border-bottom border-grey pb-2">
                                <div className="col-auto">
                                    <h2 className="section-title mb-0">My Events</h2>
                                </div>
                                <div className="col-auto data-action-bar">
                                    <button type="button" className="data-action-cta shadow-sm"><i className="ti-filter"></i></button>
                                </div>
                            </div>
                            <div className="position-relative">
                                <div className="row justify-content-start">
                                    {eventsList.map((item) => (
                                        <div className="col-md-6 col-lg-4 mb-4" key={item.id}>
                                            <EventCard eventItem={item} dispatch={dispatch} deleteEventAction={deleteEventAction} getEvents={getEvents} token={token} isShared={false} />
                                        </div>
                                    ))}
                                </div>
                                {pageCount > 1 && (
                                    <div className="pt-4">
                                        <ReactPaginate
                                            previousLabel="<"
                                            nextLabel=">"
                                            pageClassName="page-item"
                                            pageLinkClassName="page-link"
                                            previousClassName="page-item"
                                            previousLinkClassName="page-link"
                                            nextClassName="page-item"
                                            nextLinkClassName="page-link"
                                            breakLabel="..."
                                            breakClassName="page-item"
                                            breakLinkClassName="page-link"
                                            pageCount={pageCount}
                                            activeLinkClassName=""
                                            disabledLinkClassName="bg-light"
                                            marginPagesDisplayed={2}
                                            pageRangeDisplayed={5}
                                            onPageChange={handlePageChange}
                                            containerClassName="pagination d-flex"
                                            activeClassName="active"
                                            forcePage={pageOffset}
                                        />
                                    </div>
                                )}
                            </div>
                        </>
                    ) : (
                        <div className="text-start">
                            <p>No events found.</p>
                        </div>
                    )}
                </>
            )}
        </>
    );
};

export default MyEventsPage;
